/******************************************************************************************
 *
 *		Overlay
 *		-----------------------------------------------------------------------------------		
 *		Developer: Rogerio Pinto
 *      Github: https://github.com/rogeriopinto/overlay
 *		Date: 09-11-2011
 *		
 ***/

/*
	Commits are at the bottom of this
 */


// Make sure helper functions have been attached
if( !_app
	|| !_app.winHeight
	|| !_app.winWidth 
	|| !_app.elementRealWidth 
	|| !_app.elementRealHeight 
	|| !_app.addEvent 
	|| !_app.removeEvent 
	|| !_requestAnimationFrame 
	|| !_cancelAnimationFrame ) {

		throw new Error( 'Helper functions not defined.' );
}



Overlay = (function(){
	
	return function(settings){
		
	//--Priviledged properties
	
		if( !settings ) var settings = {};
	
		this.settings = {};
		this.settings.minTop			= (typeof settings.minTop !== 'undefined') ? settings.minTop : 50;
		this.settings.fallbackId		= (typeof settings.fallbackId !== 'undefined') ? settings.fallbackId : 'fallback';
		this.settings.overlayId			= (typeof settings.overlayId !== 'undefined') ? settings.overlayId : 'overlay';
		this.settings.duration			= (typeof settings.duration !== 'undefined') ? settings.duration : 400;
		this.settings.hasCloseBtn		= (typeof settings.hasCloseBtn !== 'undefined') ? settings.hasCloseBtn : true;
		this.settings.closeBtnClass		= (typeof settings.closeBtnClass !== 'undefined') ? settings.closeBtnClass : 'close';
		this.settings.closeBtnText		= (typeof settings.closeBtnText !== 'undefined') ? settings.closeBtnText : 'x';
		this.settings.hasFallbackClick	= (typeof settings.hasFallbackClick !== 'undefined') ? settings.hasFallbackClick : true;
		

		// Set up fallback
		this.fallback = document.getElementById(this.settings.fallbackId);
		if(!this.fallback)
		{
			this.fallback = document.createElement('div');
			this.fallback.setAttribute('id',this.settings.fallbackId);
			this.fallback.style.display = 'none';
			
			if( typeof this.fallback.style.opacity === 'string' )
				this.fallback.style.opacity = '0';
			else
				this.fallback.style.filter = 'alpha(opacity=0)';
			
			document.body.appendChild( this.fallback );
		}
		
		
		// Set up overlay
		this.overlay = document.getElementById(this.settings.overlayId);
		if(!this.overlay)
		{
			this.overlay = document.createElement('div');
			this.overlay.setAttribute('id',this.settings.overlayId);
			this.overlay.style.display = 'none';
			
			if( typeof this.overlay.style.opacity === 'string' )
				this.overlay.style.opacity = '0';
			else
				this.overlay.style.filter = 'alpha(opacity=0)';			
			
			document.body.appendChild( this.overlay );
		}
		
		
		// Set overlay to the off state
		this.is_on = false;
		
	}; // Return

})();


Overlay.prototype = (function(){
	
	
// Private methods -----------------------------------------------------------------------------------------------------------------------------

	// Animate ease functions
	// http://javascript.info/tutorial/animation
	function makeEaseInOut(delta) {  
	  return function(progress) {
	    if (progress < .5)
	      return delta(2*progress) / 2;
	    else
	      return (2 - delta(2*(1-progress))) / 2;
	  };
	}
	
	function quad(progress) {
	  return Math.pow(progress, 2);
	}

	var animationFunc = makeEaseInOut(quad);	
	
	
	// Animates opacity of overlay and fallback
	function animateOpacity( animationDirection, callback )
	{
		var self = this,													// Reference to self
			startTime = +new Date(),										// Start time of animation
			endTime = startTime + this.settings.duration;					// End time of animation
			
		
		// if an animation is active cancel it first
		if(this.animationRequest)
		{
			_cancelAnimationFrame(this.animationRequest);
			this.animationRequest = null;
		}
		
		
		// Set function to run on each step of the animation
		var step = function(){
			
			var current = +new Date(),								// Current time of the animation
				remaining = endTime - current,						// Time remaining for the animation
				rate = 1 - (remaining / self.settings.duration);	// Stage of the animation - Number between 0 and 1
				
				
			// If there is less than 10 milliseconds set the stage as final	
			if(remaining <= 10) rate = 1; 
			
			// Calculate position of the element based on current stage
			var opct = animationFunc(rate);
			
			// If it is hiding the overlay
			if( animationDirection === 0 ) opct = -opct + 1; // Set opacity in the opposite direction
			
			// Set opacity in all elements
			if( typeof self.overlay.style.opacity === 'string' )
			{
				self.overlay.style.opacity = opct;
				self.fallback.style.opacity = opct;
			}
			else
			{
				self.overlay.style.filter = 'alpha(opacity=' + (opct * 100) + ')';
				self.fallback.style.filter = 'alpha(opacity=' + (opct * 100) + ')';
			}	
			
			// If it is not the end of the animation
			if(remaining >= 10)
			{
				self.animationRequest = _requestAnimationFrame( step );	
			}
			else // If it is less than 10 milliseconds to the end
			{
				// And the overlay is being hidden
				if( animationDirection === 0 )
				{
					self.overlay.style.display = 'none';
					self.fallback.style.display = 'none';					
				}
				
				if( typeof callback === 'function' ) callback.call( self );
			}
		};
		
		step();	
		
	}	
	
	
// RETURN ------------------------------------------------------------------------------------------------------------	
	
	
	return {
	
// Resizes the overlay	
		resize: function(settings)
		{
			if( !settings ) var settings = {};
			if( !settings.width ) settings.width = _app.elementRealWidth( this.overlay );
			if( !settings.height ) settings.height = _app.elementRealHeight( this.overlay );				
			if( !settings.callback ) settings.callback = null;	
			
			this.overlay.style.width = settings.width + 'px';
			this.overlay.style.height = settings.height + (isNaN(settings.height) ? '' : 'px');
			
			if(settings.callback) settings.callback.call( this );
			
			return this.position({
				width: settings.width,
				height: settings.height
			});
		},
		
		
// Positions the overlay		
		position: function(settings)
		{
			if( !settings ) var settings = {};
			if( !settings.width ) settings.width = this.overlay.offsetWidth;
			if( !settings.height ) settings.height = this.overlay.offsetHeight;			
			if( !settings.callback ) settings.callback = null;
					
			// calculate the top position of  the overlay
			var top = (_app.winHeight() / 2) - (settings.height / 2);
			if( top < this.settings.minTop ) top = this.settings.minTop;
			
			this.overlay.style.top = top + 'px';
			this.overlay.style.left = '50%';
			this.overlay.style.marginLeft = '-' + Math.round(settings.width / 2) + 'px';
			
			if(settings.callback) settings.callback.call( this );
			
			return this;
		},
		
		
// Displays the overlay		
		show: function(settings)
		{
			if( this.is_on ) return this;
			
			this.is_on = true;
			
			var self = this;
			
			if( !settings ) var settings = {};
			if( !settings.animate ) settings.animate = true;
			if( !settings.callback ) settings.callback = null;
			
			// Reset overlay and fallback
			this.overlay.style.display = 'block';
			this.fallback.style.display = 'block';	
			if( typeof this.overlay.style.opacity === 'string' )
			{
				this.overlay.style.opacity = 0;	
				this.fallback.style.opacity = 0;
			}		
			else
			{
				this.overlay.style.filter = 'alpha(opacity=0)';	
				this.fallback.style.filter = 'alpha(opacity=0)';	
			}
			
			this.overlay.style.position = 'fixed';
			
			this.fallback.style.position = 'fixed';
			this.fallback.style.top = 0;
			this.fallback.style.left = 0;	
			this.fallback.style.width = _app.winWidth() + 'px';
			this.fallback.style.height = _app.winHeight() + 'px';				
			
			// Add fallback click
			if( this.settings.hasFallbackClick )
				this.addFallbackClick();
				
			// Window resize callback
			this.onWindowResize = function(){
				self.fallback.style.width = _app.winWidth() + 'px';
				self.fallback.style.height = _app.winHeight() + 'px';				
				self.position();
			};	
			
			_app.addEvent( window, 'resize', this.onWindowResize, false );
			
			_app.addEvent( window, 'orientationchange', this.onWindowResize, false );
			
			// Position overlay and fallback
			this.position();			
							
			// Animate items in
			animateOpacity.call( this, 1, function(){
				
				// If the close button is enabled add it
				if( self.settings.hasCloseBtn ) self.addCloseBtn();
				
				// Run callback
				if( typeof settings.callback === 'function' ) settings.callback.call( self );
				
			} );
			
			return this;
		},
		
		
// Hides the overlay		
		hide: function(settings)
		{
			if( !this.is_on ) return this;
			
			this.is_on = false
			
			if( !settings ) var settings = {};
			if( !settings.animate ) settings.animate = true;
			if( !settings.callback ) settings.callback = null;
			
			// Remove close button
			this.removeCloseBtn();
			
			// Remove fallback click
			this.removeFallbackClick();
			
			_app.removeEvent( window, 'resize', this.onWindowResize );
			
			_app.removeEvent( window, 'orientationchange', this.onWindowResize );	
			
			// Animate items in
			animateOpacity.call( this, 0, settings.callback );	
					
			return this;
		},
		
		
// Inserts content		
		content: function(settings)
		{
			if( !settings ) var settings = {};
			if( !settings.html ) settings.html = '';
			if( !settings.callback ) settings.callback = null;
			
			if( typeof settings.html === 'string' )
			{
				this.overlay.innerHTML = settings.html;
			}
			else
			{
				this.overlay.innerHTML = '';
				this.overlay.appendChild( settings.html );	
			}
			
			if( typeof settings.callback === 'function' ) settings.callback.call( this );
			
			return this;
		},
		

		
// Enables the close butto		
		enableCloseBtn: function()
		{
			this.settings.hasCloseBtn = true;
			return this;
		},
		
		
// Disables the close button		
		disableCloseBtn: function() 
		{
			this.settings.hasCloseBtn = false;
			return this;
		},		
		
		
// Adds the close button		
		addCloseBtn: function()
		{
			var self = this; // Reference to self
			
			// Create anchor tag and attach it to overlay
			var a = document.createElement( 'a' );
			a.setAttribute( 'href', '#' );
			a.className = this.settings.closeBtnClass;
			a.appendChild( document.createTextNode(this.settings.closeBtnText) );
			this.overlay.appendChild( a );
			
			// Function to handle click event of anchor
			function onClick( e )
			{
				// Get event object
				if( !e ) var e = window.event;
				
				self.hide();
				
				// Prevent default action
				if( e.preventDefault )
					e.preventDefault();
				else
					e.returnValue = false;		
			}
			
			_app.addEvent( a, 'click', onClick, false );
				
			return this;	
		},
		
		
// Removes the close button		
		removeCloseBtn: function()
		{
			// If there is support for 'querySelector'
			if( document.querySelector )
			{
				// Get element and remove it
				var closeBtn = this.overlay.querySelector( '.' + this.settings.closeBtnClass );
				if( closeBtn ) 
					closeBtn.parentNode.removeChild( closeBtn );
			}
			else // Otherwise
			{
				// Get all children of overlay
				var overlayChildren = this.overlay.getElementsByTagName('*');
				
				// For each child	
				for( var i = 0; i < overlayChildren.length; i++ )
				{
					// Grab classes as array (Remove spaces from both ends and split it into an array of strings)
					var classes = overlayChildren[i].className.replace(/^\s+|\s+$/g, '').split(' ');
					
					// For each class name
					for( var x = 0; x < classes.length; x++ )
					{
						// If it matches the button's class name
						if( classes[x] === this.settings.closeBtnClass )
						{
							// Remove it
							overlayChildren[i].parentNode.removeChild( overlayChildren[i] );
							return;	
						}
					}
				}	
			}
			return this;
		},
		
		
// Enables fallback click		
		enableFallbackClick: function()
		{
			this.settings.hasFallbackClick = true;
			return this;
		},
		
		
// Disables fallback click
		disableFallbackClick: function() 
		{
			this.settings.hasFallbackClick = false;
			return this;
		},
		
				
		
// Adds fallback click
		addFallbackClick: function()
		{
			var self = this;
			
			this.onFallbackClick = function(){
				self.hide();
			};
			
			_app.addEvent( this.fallback, 'click', this.onFallbackClick, false );
			
			return this;	
		},
		
		
// Removes fallback click
		removeFallbackClick: function()
		{
			if( typeof this.onFallbackClick !== 'function' ) return;
			
			_app.removeEvent( this.fallback, 'click', this.onFallbackClick );
			
			return this;	
		}
		
		
	} // return {

})();


/*
	COMMITS
	
		17/03/2014	-	Ammended 'hide()' to check if overlay is on before hiding
		17/03/2014	-	Added property 'addCloseBtn' to 'content()' function
		17/03/2014	-	In 'resize()' added check to set height as auto
		
		24/03/2014 	-	Updated the jQuery library to the latest version
		
		
		Started development: 04/09/2014
		
		07/09/2014 - 	Replaced some jQuery functions with native javascript
						Removed support to setting parent via the 'parent' attribute during instantiation
						Added new functions to api: enableFallbackClick, disableFallbackClick, addFallbackClick, removeFallbackClick, enableCloseBtn, disableCloseBtn, addCloseBtn, removeCloseBtn
						
		12/09/2014 - 	Replaced some jQuery functions with native javascript	
		
		09/10/2014 -    Added external helper/polyfills and deleted unused files and scripts					
		
*/