"use strict";

var defaultScrollerCSS = {
	"div": {
	},

	"image": {
		"border": "1px solid white"
	},
};

var defaultScrollerItemsCSS = {
	"img": {
		"opacity": "0.7",
		"cursor": "pointer",
		"border": "1px solid black",
	},

	"img.current": {
		"opacity": "1 !important",
		"border": "1px solid white !important",
	},

	"div": {
		"width": "15px",
		"height": "15px",
		"cursor": "pointer",
		"border-radius": "50%",
		"background-color": "white",
		"border": "2px solid black",
	},

	"div.current" : {
		"background-color": "black !important",
		"border": "2px solid white !important"
	},
};

function Slidon (element, options) {
	var self = this;

	element = this.filter(element);
	var items = element.children("*");

	this.config = {
		id: slidonID(),
		tags: items.eq(0).prop("tagName"),
		options: this.validateOptions(options),
	};

	/* Validate Slidon elements */

	items.each(function(index, element) {
		if (self.config.options.scroller.enabled && self.config.options.scroller.proportion && (jQuery(this).prop("tagName") != "IMG")) {
			throw new Error("Slidon initialization error! proportion option is only supported on image Slidons and at least one of your children elements is not an image");
		}
	});

	if (items.length < 2) {
		throw new Error("Slidon initialization error! needed at least two elements inside Slidon to be initialized");
	}

	if (items.length < this.config.options["display"]) {
		throw new Error('Slidon initialization error! specified ' + this.config.options["display"] + ' "display" items and you only have ' + items.length + ' elements inside your Slidon to slide');
	}

	if (items.length < this.config.options["iter"]) {
		throw new Error('Slidon initialization error! specified ' + this.config.options["iter"] + ' "iter" items and you only have ' + items.length + ' elements inside your Slidon to slide');
	}

	initElements(items, this.config.options.type, this.config.options.forceStyling);

	/* Init internal symbols */

	this.config.queue = [];
	this.config.force = false;

	this.config.lastEffect = "linear";
	this.config.attributes = element[0].attributes;

	this.config.buttons = {};
	this.config.position = 0;
	this.config.working = true;
	this.config.seen = currentTime();
	this.config.items = items.length;

	this.config.options.autoScroll.stop = true;
	this.config.options.autoScroll.handler = null;
	this.config.options.autoScroll.bounceStop = false;

	if (this.config.options.carousel) {
		this.up = this.moveLeftCarousel;
		this.left = this.moveLeftCarousel;
		this.down = this.moveRightCarousel;
		this.right = this.moveRightCarousel;
		this.config.options.autoScroll.bounce = false;
	} else {
		this.up = this.moveLeft;
		this.left = this.moveLeft;
		this.down = this.moveRight;
		this.right = this.moveRight;
	}

	/* 
	 * Slidon elements resolution might not be available in different scenarios, in
	 * such cases, we will manually render Slidon outside of the client view-port and
	 * wait till browser sets all elements final resolutions.
	 *
	 */

	element.attr({
		"data-slidon": this.config.id,
		"data-slidon-init": 0
	});

	this.config.element = element;

	if (!items.width() || !items.height()) {
		this.styling = {
			left: element.css("left"),
			display: element.css("display"),
			position: element.css("position"),
		};

		jQuery("<div></div>").attr("data-slidon", this.config.id).addClass("slidon-position").css({
			width: 0,
			height: 0,
			display: "none",
			position: "absolute"
		}).insertBefore(element);

		element.appendTo("body").css({
			"left": -9999999,
			"display": "block",
			"position": "absolute",
		});

		this.initTries = 0;

		setTimeout(function() {
			waitRenderingResolutions.apply(self);
		}, 10);

	} else {
		this.config.width = items.outerWidth(true);
		this.config.height = items.outerHeight(true);

		initSlidon.apply(self);
	}
}

/* Private Methods */ 

function slidonID () {
	do {
		var randomID = Math.floor(Math.random() * 100000) + 1;
	} while(jQuery("[data-slidon=" + randomID + "]").length);
	
	return randomID;
}

function getContainer(attributes) {
	var container = jQuery("<div></div>");

	jQuery.each(attributes, function() {
		container.attr(this.name, this.value);
	});	

	return container;
}

function initElements (elements, type, forceStyling) {
	type = (type == "horizontal") ? "inline-block" : "block";

	elements.each(function(index, element) {
		jQuery(this).attr("data-slidon-index", index + 1).css({
			"display": type,
			"position": "relative",
			"overflow": "hidden",
		}).addClass("slidon-item");

		/* 
		 * NOTE: When adding elements into Slidon slider container div we might be 
		 * breaking elements styling if they were selected through parent containers. 
		 * To avoid styling headaches and trying to make Slidon as less invasive as 
		 * possible, we enable an option that if specified will apply each elements 
		 * styling inline so Slidon internal structure does not break its styling.
		 *
		 */

		if (forceStyling) {
			jQuery(this).css(elementCSS(jQuery(this))).find("*").each(function(index, element) {
				jQuery(this).css(elementCSS(jQuery(this)));
			});
		}
	});

	return elements;
}

function renderElement(element) {
	element.addClass("slidon-render slidon-remove").css({
		left: -9999999,
		position: "absolute",

	}).appendTo(jQuery("body"));

	return element;
}

function positionSlidon () {
	var element = this.config.element;
	var position = this.filter(jQuery("[data-slidon=" + this.config.id + "].slidon-position"));

	/* 
	 * NOTE: If position is missing, probably the container where the Slidon
	 * was located when initialized was removed while Slidon was still trying
	 * to initialize, if not something really strange happened. Anyway, we
	 * abort initialization as i don't know were to locate back the Slidon and
	 * i report warning in console so client is aware of this.
	 *
	 **/

	if (!position.length) {
		element.remove();
		throw Error("Slidon initializing error! Could not find Slidon container, probably it was removed from the DOM");
	}

	element.css({
		left: this.styling.left,
		display: this.styling.display,
		position: this.styling.position,

	}).insertAfter(position);

	position.remove();
	delete this.styling;
}

function waitRenderingResolutions () {
	var element = this.config.element;
	var items = element.children("*");

	if (++this.initTries > 100) {
		positionSlidon.apply(this);
		throw new Error("Slidon initialization error! couldn't calculate sliding elements width/height, kindly try styling their width/height");
	}

	var XY = {
		width: items.eq(0).width(),
		height: items.eq(0).height()
	};

	var RENDER_READY = true;

	/*
	 * NOTE: When client uses the operating systems fonts increasing feature, rendering dimensions 
	 * of images seems to get kinda crazy, that's why i'm doing this "strage" +/- pixels check. Next
	 * final version will handle different DIVs sizes which will handle this automatically.
	 * 
	 */

	items.each(function(INDEX, ELEMENT) { 
		if (!jQuery(this).width() || !jQuery(this).height() ||
				(jQuery(this).width() > (XY.width + 5)) || 
				(jQuery(this).width() < (XY.width - 5)) || 
				(jQuery(this).height() > (XY.height + 5)) || 
				(jQuery(this).height() < (XY.height - 5))) {

			RENDER_READY = false;
			return false; // break
		}
	});

	var self = this;

	if (!RENDER_READY) {
		setTimeout(function() {
			waitRenderingResolutions.apply(self);
		}, 100);

	} else {
		this.config.width = items.outerWidth(true);
		this.config.height = items.outerHeight(true);

		positionSlidon.apply(this);
		initSlidon.apply(this);
	}
}

function error(error) {
	console.log(error);
	return this;
}

function currentTime() {
	return new Date().getTime();
}

function initSlidon () {
	var self = this;
	var scroller = null;
	var slidon = this.config.element;
	var items = slidon.children("*");

	/* Build Slidon structure */

	var slider = jQuery('<div class="slidon-slider"></div>');
	var slidonContainer = jQuery('<div class="slidon-container"></div>');

	slidonContainer.attr({
		"data-slidon": this.config.id,
		"data-slidon-current": 1,
		"data-slidon-direction": this.config.options.direction
	});

	items.detach().first().addClass("slidon-current");
	items.appendTo(slider);

	slider.appendTo(slidonContainer);
	slidonContainer.appendTo(slidon);

	var slidonCSS = {
		"overflow": "hidden",
	};

	var slidonContainerCSS = {
		"overflow": "hidden",
		"display" : "inline-block",
		"position": "relative"
	}

	var sliderCSS = {
		"position": "relative"
	};

	if (this.config.options.type == "horizontal") {
		slidonCSS["height"] = this.config.height + "px";
		slidonCSS["width"] = (this.config.width * this.config.options.display) + "px";

		slidonContainerCSS["height"] = this.config.height + "px";
		slidonContainerCSS["width"] = (this.config.width * this.config.options.display) + "px";

		sliderCSS["width"] = (this.config.width * items.length) + "px";
		sliderCSS["height"] = this.config.height + "px";
	} else {
		slidonCSS["height"] = (this.config.height * this.config.options.display) + "px";
		slidonCSS["width"] = this.config.width + "px";

		slidonContainerCSS["height"] = (this.config.height * this.config.options.display) + "px";
		slidonContainerCSS["width"] = this.config.width + "px";

		sliderCSS["width"] = this.config.width + "px";
		sliderCSS["height"] = (this.config.height * items.length) + "px";
	}

	slider.css(sliderCSS);
	slidonContainer.css(slidonContainerCSS);

	jQuery.each(slidonCSS, function(key, value) {
		slidon.css(key, value);
	});

	/* Queue initial index, should be done before calling ready() callback in next version */

	if (this.config.options.index > 1) {
		this.index("slidon", this.config.options.index, 0);
	}
	
	/* Buttons initialization */

	if (this.config.options.buttons) {
		var buttons = ["left", "right", "up", "down", "next", "prev", "invert", "shuffle", "stop", "start", "show", "hide"];

		jQuery.each(buttons, function(index, button) {
			if (self.config.options.buttons[button]) {
				self.button("slidon", button, self.config.options.buttons[button]);
			}
		})

		this.config.options.buttons = {};
	}

	/* 
	 * Scrolling box initialization, i want to keep track of the scroller box size
	 * at all times as this will enable me (in the next version :P) to properly re
	 * locate it based on size and detect when it overflows the Slidon view to for
	 * example initialize a smaller Slidon representing the Scrolling Box. Anyway,
	 * usually the size of the scrolling box won't be rendered in the moment we
	 * generate it so i'm finishing the initialization process (finishSlidon) after
	 * the Scrolling Box is initialied properly. If no Scrolling Box enabled, we go
	 * directly to finishSlidon().
	 *
	 **/

	if (this.config.options.scroller.enabled) {
		createScroller(this, items);
	} else {
		finishSlidon.apply(this);
	}
}

function finishSlidon(slidon) {
	var self = this;
	var slidon = this.config.element;

	delete this.config.element;

	/* "Auto Scrolling" initialization */

	if (this.config.options.autoScroll.enabled) {
		if (parseInt(this.config.options.autoScroll.time) < 30) {
			this.config.options.autoScroll.time = 30; // Need to fix this!
		}

		slidon.off("mouseenter.slidon").on("mouseenter.slidon", function() {
			self.mouseIn("user");
		});

		slidon.off("mouseleave.slidon").on("mouseleave.slidon", function() {
			self.mouseOut("user");
		});

		this.start("autoScroll");
	}

	if (this.config.options.callbacks.ready) {
		this.config.options.callbacks.ready();
	}

	this.config.working = false;
	slidon.attr("data-slidon-init", 1);

	this.processQueue();
}

function elementCSS(a) {
	var sheets = document.styleSheets,
		o = {};

	for (var i in sheets) {
		var rules = sheets[i].rules || sheets[i].cssRules;
		for (var r in rules) {
			if (a.is(rules[r].selectorText)) {
				o = jQuery.extend(o, JSONCSS(rules[r].style), JSONCSS(a.attr('style')));
			}
		}
	}

	return o;
}

function JSONCSS(css) {
	var s = {};
	if (!css) return s;
	if (css instanceof CSSStyleDeclaration) {
		for (var i in css) {
			if ((css[i]).toLowerCase) {
				s[(css[i]).toLowerCase()] = (css[css[i]]);
			}
		}
	} else if (typeof css == "string") {
		css = css.split("; ");
		jQuery.each(css, function(i) {
			var l = css[i].split(": ");
			s[l[0].toLowerCase()] = (l[1]);
		});
	}
	return s;
}

function shuffleArray(ARRAY) {
	for (var j, x, i = ARRAY.length; i; j = Math.floor(Math.random() * i), x = ARRAY[--i], ARRAY[i] = ARRAY[j], ARRAY[j] = x);
	return ARRAY;
}

function scrollerStyling(position) {
	var styling = {
		"left": {
			"left": 0
		},
		"right": {
			"right": 0
		},
		"center": {
			"left": 0,
			"right": 0,
			"margin-left": "auto",
			"margin-right": "auto"
		},

		"top": {
			"top": 0
		},
		"middle": {
			"top": "50%",
			"transform": "translateY(-50%)"
		},
		"bottom": {
			"bottom": 0
		},
	};

	return jQuery.extend(styling[position.split(" ")[0]], styling[position.split(" ")[1]]);
}

function scaleImage(image, width, height, proportion) {
	var element = image.clone(true);

	element.width(Math.ceil(width * +("0." + proportion)));
	element.height(Math.ceil(height * +("0." + proportion)));

	return element.removeClass("slidon-item slidon-current");
}

function typeDirection(direction, type) {
	var typeDirections = {
		"right": {
			"vertical": "down",
			"horizontal": "right",
		},

		"down": {
			"vertical": "down",
			"horizontal": "right",
		},

		"up": {
			"vertical": "up",
			"horizontal": "left",
		},

		"left": {
			"vertical": "up",
			"horizontal": "left",
		}
	};

	return typeDirections[direction][type];
}

function invertDirection(direction) {
	var invertedDirections = {
		"up": "down",
		"down": "up",
		"left": "right",
		"right": "left"
	};

	return (invertedDirections[direction] || "right");
}

function directionMethod(direction) {
	var directionMethods = {
		"up": "left",
		"left": "left",
		"down": "right",
		"right": "right"
	};

	return (directionMethods[direction] || "right");
}

function injectStylingClass(selector, style) {
	var styleCSS = '';

	jQuery.each(style, function(key, value) {
		styleCSS += key + ": " + value + ";\n";
	});

	jQuery('<style type="text/css">' + selector + " {\n" + styleCSS + "}\n</style>").appendTo("head");
}

function createScrollerElements(slidon, elements) {
	var width = slidon.config.width;
	var height = slidon.config.height;
	var proportion = slidon.config.options.scroller.proportion;

	var items = jQuery([]);
	var CSStype = (slidon.config.options.scroller.type == "horizontal") ? "inline-block" : "block";

	elements.each(function(index) {
		var element = (proportion) ? scaleImage(jQuery(this), width, height, proportion) : jQuery('<div></div>');

		element.addClass("slidon-scroller-item").attr("data-slidon-index", index + 1).css({
			"display": CSStype,
			"position": "relative",
		});

		items = items.add(element);
	});

	/*
	 * NOTE: I create an exact representation of the Slidon container
	 * and check if client styled the "slidon-scroller-item" class, if
	 * he didn't i inject our default styling class. Please note that
	 * instead of applying inline the CSS i inject a styling class in
	 * order to support the styling effect when passing the "slidon-current" 
	 * through the Scrolling Box elements when iterating. Also note 
	 * that the class injected is specific for each Slidon without
	 * styling present so you can style your different Slidons as you
	 * wish using descendant selectors from within your Slidons 
	 * containers and the default styling will be applied only on those 
	 * you didn't style. I'm not sure if what i just said actually
	 * was written properly to make sense BUT i won't write it again :P
	 *
	 **/

	var container = getContainer(slidon.config.attributes);
	jQuery('<div class="slidon-container"><div class="slidon-scroller"></div></div>').appendTo(container);
	var scroller = container.children().children();

	if (proportion) {
		jQuery('<img class="slidon-scroller-item slidon-remove">').appendTo(scroller);
		var itemCSS = elementCSS(scroller);

		if (!Object.keys(itemCSS).length) {
			var slidonSelector = '[data-slidon="' + slidon.config.id + '"].slidon-container > .slidon-scroller > .slidon-scroller-item';

			injectStylingClass(slidonSelector, defaultScrollerItemsCSS["img"]);
			injectStylingClass(slidonSelector + ".slidon-current", defaultScrollerItemsCSS["img.current"]);
		}

	} else { 
		jQuery('<div class="slidon-scroller-item slidon-remove"></div>').appendTo(scroller);
		var itemCSS = elementCSS(scroller);

		if (!Object.keys(elementCSS(jQuery('<div class="slidon-scroller-item"></div>'))).length) {
			var slidonSelector = '[data-slidon="' + slidon.config.id + '"].slidon-container > .slidon-scroller > .slidon-scroller-item';

			injectStylingClass(slidonSelector, defaultScrollerItemsCSS["div"]);
			injectStylingClass(slidonSelector + ".slidon-current", defaultScrollerItemsCSS["div.current"]);
		}
	}

	items.off("click.slidon").on("click.slidon", function() {
		slidon.index("user", parseInt(jQuery(this).attr("data-slidon-index")));
	});

	return items;
}

function createScroller(slidon, elements) {
	slidon.config.scrollerTries = 0;
	slidon.config.scrollerContainer = getContainer(slidon.config.attributes).css({
		width: "",
		height: ""
	});
	
	jQuery('<div class="slidon-container" data-slidon="' + slidon.config.id + '"></div>').appendTo(slidon.config.scrollerContainer);
	jQuery('<div class="slidon-scroller"></div>').appendTo(slidon.config.scrollerContainer.children());

	var items = createScrollerElements(slidon, elements);

	items.first().addClass("slidon-current");
	items.appendTo(slidon.config.scrollerContainer.children().children());

	renderElement(slidon.config.scrollerContainer);

	setTimeout(function() {
		createScrollerFinish(slidon);
	}, 100);
}

function createScrollerFinish(slidon) {
	if (slidon.config.scrollerTries++ > 100) {
		slidon.config.scrollerContainer.remove();
		error("Slidon Scrolling Box initializing error! Could not determine scrolling box buttons resolution. Don't forget to set fixed slidon-scroller-item width/height if you are re-styling its default style");

		delete slidon.config.scrollerTries;
		delete slidon.config.scrollerContainer;

		return finishSlidon.apply(slidon);
	}

	var scroller = slidon.config.scrollerContainer.children().children();
	var items = scroller.children();

	if (!items.width() || !items.height() ||
			(slidon.config.options.scroller.type == "horizontal" && (scroller.width() < (items.width() * items.length))) || 
				(slidon.config.options.scroller.type == "vertical" && (scroller.height() < (items.height() * items.length)))) {

		return setTimeout(function() {
			createScrollerFinish(slidon);
		}, 100);
	}

	slidon.config.options.scroller.width = scroller.width() + 1;
	slidon.config.options.scroller.height = scroller.height() + 1;

	scroller = scroller.detach();

	scroller.width(slidon.config.options.scroller.width)
	scroller.height(slidon.config.options.scroller.height);

	if (slidon.config.options.scroller.position) {
		scroller.css(scrollerStyling(slidon.config.options.scroller.position));
	} else if (!Object.keys(elementCSS(scroller)).length) {
		scroller.css("bottom center");
	}

	scroller.css({
		"z-index": "9999",
		"position": "absolute",
		"overflow": "hidden",
	});

	if (slidon.config.options.scroller.hidden) {
		scroller.css("display", "none");
	}

	scroller.prependTo(slidon.getSlidon());
	slidon.config.scrollerContainer.remove();

	delete slidon.config.scrollerTries;
	delete slidon.config.scrollerContainer;

	finishSlidon.apply(slidon);
}

/* "Private Methods", a.k.a. not expected to be overwritten but if you wish.. be my guest */

/*
 * This function is called in ALMOST all actions and checks if
 * there is an iteration currently being performed or other
 * actions queued, if true, the action gets queued and returns
 * false. If the action is "readyToRun", it returns true. This
 * function will also check if there was a callback set for the
 * action from which it was called and executes it, if the
 * callback returns false, then the function returns also false
 * as remember that you can prevent from any callback the action
 * completition by returning false;
 *
 **/

Slidon.prototype.readyToRun = function(command, args, dontRunCallback) {
	var ready = true;

	if (this.config.working || (this.config.queue.length && !this.config.force)) {
		this.config.queue.push([command, args]);
		ready = false;

	} else {
		this.config.force = false;

		if (!dontRunCallback && this.config.options.callbacks[command] && !this.config.options.callbacks[command].apply(this, args)) {
			ready = false;
		}
	}

	return ready;
}

/*
 * This is important to have in mind! when using carousel Slidons, 
 * in order to support its effect i clone some elements when the
 * iteration starts and remove them when it finishes. The thing is
 * that if for example the user had a Slidon inside a carousel Slidon
 * and the Slidon inside the Slidon tries to select itself to perform
 * any of its actions it might end up selecting the one that is about
 * to be removed generating some horrible raise conditions. In order
 * to avoid this, the class "slidon-remove" is added on the elements
 * that are about to be removed when the iteration of the carousel
 * Slidon ends and the filter function is used to filter out those
 * elements that are inside a container with the "slidon-remove" 
 * class which basically means its the cloned element about to be
 * removed. 
 *
 * SO! When selecting elements inside a carousel Slidon, ALWAYS
 * call the filter function over those elements and you will be safe.
 *
 * If retrieving a set of elements inside a carousel Slidon directly
 * using jQuery, you MUST do this and you will be safe:
 *
 * var elements = $(".slidon").find("your-elements").slidon("filter");
 *
 * TODO: Set some sort of transparent hook overloading some core
 * jQuery methods to make this filtering without any user interaction.
 *
 **/

Slidon.prototype.filter = function(elements) {
	var slidon = false;

	/* NOTE: This should be done using :not() selector */

	elements.each(function() {
		if (!jQuery(this).parents(".slidon-remove").length) {
			slidon = jQuery(this);
		}
	});

	return slidon;
}

Slidon.prototype.getSlidon = function() {
	return this.filter(jQuery("[data-slidon=" + this.config.id + "].slidon-container"));
}

Slidon.prototype.getSlider = function() {
	return this.getSlidon().children(".slidon-slider");
}

Slidon.prototype.getScroller = function() {
	return this.getSlidon().children(".slidon-scroller");
}

Slidon.prototype.id = function() {
	return this.getSlidon().attr("data-slidon");
}

Slidon.prototype.easingEffects = [
	"linear", 
	"swing", 
	"easeInQuad", 
	"easeOutQuad", 
	"easeInOutQuad", 
	"easeInCubic", 
	"easeOutCubic", 
	"easeInOutCubic", 
	"easeInQuart", 
	"easeOutQuart", 
	"easeInOutQuart", 
	"easeInQuint", 
	"easeOutQuint", 
	"easeInOutQuint", 
	"easeInExpo", 
	"easeOutExpo", 
	"easeInOutExpo", 
	"easeInSine", 
	"easeOutSine", 
	"easeInOutSine", 
	"easeInCirc", 
	"easeOutCirc", 
	"easeInOutCirc", 
	"easeInElastic", 
	"easeOutElastic", 
	"easeInOutElastic", 
	"easeInBack", 
	"easeOutBack", 
	"easeInOutBack", 
	"easeInBounce", 
	"easeOutBounce", 
	"easeInOutBounce"
];

Slidon.prototype.randomEffect = function() {
	do {
		var effect = this.easingEffects[Math.floor(Math.random() * (this.easingEffects.length - 1))];
	} while(effect.substr(-3) == this.config.lastEffect.substr(-3));

	return effect;
}

Slidon.prototype.getCurrentElement = function() {
	return parseInt(this.getSlidon().attr("data-slidon-current"));
}

Slidon.prototype.setCurrentElement = function(index) {
	var slider = this.getSlider();

	slider.children(".slidon-item.slidon-current").removeClass("slidon-current");
	slider.children("[data-slidon-index=" + index + "].slidon-item").addClass("slidon-current");

	if (this.config.options.scroller.enabled) {
		var scroller = this.getScroller();

		scroller.children(".slidon-scroller-item.slidon-current").removeClass("slidon-current");
		scroller.children("[data-slidon-index=" + index + "].slidon-scroller-item").addClass("slidon-current");
	}

	this.getSlidon().attr("data-slidon-current", parseInt(index));
}

Slidon.prototype.random = function(min, max) {
	return Math.floor(Math.random() * (max - min + 1)) + min	
}

Slidon.prototype.processSpeed = function(context, iter) {
	var speed = 0;

	if ((context == "index") && this.config.options.instantIndex) {
		speed = 0;
	} else if (this.config.options.speed instanceof Array) {
		speed = (this.random(this.config.options.speed[0], this.config.options.speed[1]) / this.config.options.iter) * iter;
	} else {
		speed = (this.config.options.speed / this.config.options.iter) * iter;
	}

	return speed;
}

/* 
 * Add elements in specified index, if index is not specified elements
 * will be added to the end of the Slidon (pushed). Please remember that
 * indexes in Slidon start at 1 and not 0, so if you want to insert 
 * elements at the start of a Slidon, index must be equal to 1. Also note
 * that Slidon will always keep its view on the current element so if you
 * add any ammount of elements prior to or at the same index of the current
 * element in view, the current element index will be increased by the 
 * number of elements added and the current Slidon view will follow it.
 *
 **/

Slidon.prototype.add = function (caller, elements, index) {
	if (!this.readyToRun("add", arguments)) {
		return this;
	}

	index = (!index || (index > (this.config.items+1)) || (index < 1)) ? this.config.items + 1 : index;
	elements = initElements(elements, this.config.options.type, this.config.options.forceStyling).removeAttr("data-slidon-index").removeClass("slidon-current");

	var slider = this.getSlider();

	/* Scrolling Box present? generate scrolling box index button */

	if (this.config.options.scroller.enabled) {
		var scroller = this.getScroller();
		var scrollerItems = createScrollerElements(this, elements);
	}

	/* Add new elements to Slidon at the specified element index */

	if (index > this.config.items) {
		var current = slider.children("[data-slidon-index=" + this.config.items + "].slidon-item");

		if (!current.length) {
			elements.prependTo(slider);
		} else {
			elements.insertAfter(current);
		}

		if (scroller) {
			current = scroller.children("[data-slidon-index=" + this.config.items + "].slidon-scroller-item");

			if (!current.length) {
				scrollerItems.prependTo(scroller);
			} else {
				scrollerItems.insertAfter(current);
			}
		}

	} else {
		var current = slider.children("[data-slidon-index=" + index + "].slidon-item");

		if (!current.length) {
			elements.prependTo(slider);
		} else if (!this.config.options.carousel) {
			elements.insertBefore(current);
		} else {
			if (!slider.children().index(current)) {
				elements.appendTo(slider);
			} else {
				elements.insertBefore(current);
			}
		}

		if (scroller) {
			current = scroller.children("[data-slidon-index=" + index + "].slidon-scroller-item");

			if (!current.length) {
				scrollerItems.prependTo(scroller);
			} else {
				scrollerItems.insertBefore(current);
			}
		}
	}

	/* Set data-slidon-index values for new elements and those following them */

	for(var i = this.config.items; i >= index; i--) {
		slider.children("[data-slidon-index=" + i + "]").attr("data-slidon-index", i + elements.length);
	}

	for(i = 0; i < elements.length; i++) {
		elements.eq(i).attr("data-slidon-index", index + i);
	}

	/* Update position and slidon-current if Slidon was empty */

	if (!this.config.items) {
		slider.children(":first").addClass("slidon-current");

		if (scroller) {
			scroller.children(":first").addClass("slidon-current");
		}
	}

	if (!this.config.options.carousel) {
		this.config.position = (-1) * (parseInt(slider.children(".slidon-current").attr("data-slidon-index")) - 1);
	}

	this.config.items += elements.length;

	/* Set Scrolling Box elements data-slidon-index values */

	if (scroller) {
		scroller.children(".slidon-scroller-item").each(function(index) {
			jQuery(this).attr("data-slidon-index", index + 1);
		});

		/* TODO: fix this */

		var self = this;
		setTimeout(function() {
			var totalRes = 0;

			scroller.children().each(function() {
				totalRes += (self.config.options.scroller.type == "horizontal") ? jQuery(this).outerWidth(true) : jQuery(this).outerHeight(true);
			});

			if (self.config.options.scroller.type == "horizontal") {
				scroller.width(totalRes);
			} else {
				scroller.height(totalRes);
			}

		}, 100);
	}

	/* Fix sliding DIV position so slidon-current item remains un-touched */

	if (this.config.options.type == "horizontal") {
		slider.width(slider.width() + (this.config.width * elements.length));

		if (!this.config.options.carousel) {
			slider.css("right", (this.config.width * (-1) * this.config.position) + "px");
		}

	} else {
		slider.height(slider.height() + (this.config.height * elements.length));

		if (!this.config.options.carousel) {
			slider.css("top", (this.config.height * this.config.position) + "px");
		}
	}

	if (this.config.options.callbacks["addComplete"]) {
		this.config.options.callbacks["addComplete"](caller, elements, index);
	}

	return this;
};

Slidon.prototype.removeElements = function(element, number, slider) {
	slider = slider || this.getSlider();

	/* Remove specified elements */

	if (!this.config.options.carousel) {
		var removed = slider.children().slice(element - 1, element - 1 + number);
		var removedCurrent = removed.filter(".slidon-current").length;

		removed.remove();

	} else {
		var removedCurrent = false;

		for (var i = element; i < element + number; i++) {
			var current = slider.children("[data-slidon-index=" + i + "]");

			if (current.hasClass("slidon-current")) {
				removedCurrent = true;
			}

			current.remove();
		}
	}

	/* Update elements new data-slidon-index properties */

	for(var i = element + number; i <= this.config.items; i++) {
		slider.children("[data-slidon-index=" + i + "]").attr("data-slidon-index", i - number);
	}

	this.config.items -= number;

	/* Update sliding DIV properties */

	if (this.config.options.type == "horizontal") {
		slider.width(slider.width() - (this.config.width * number));
	} else {
		slider.height(slider.height() - (this.config.height * number));
	}

	/* If removed current Slidon element, update "slidon-current" element */

	if (removedCurrent) {
		var currentIndex = (element <= this.config.items) ? element : element - 1;
		slider.children("[data-slidon-index=" + currentIndex + "]").addClass("slidon-current");

		if (this.config.options.carousel && (currentIndex != element)) {
			slider.children().eq(-1).prependTo(slider);
		}
	}

	/* Position slider, position is always 0 on carousel sliders ;) */

	this.config.position = (this.config.items && !this.config.options.carousel) ? (-1) * (parseInt(slider.children(".slidon-current").attr("data-slidon-index")) - 1) : 0;

	if (this.config.options.type == "horizontal") {
		slider.css("right", (this.config.width * (-1) * this.config.position) + "px");
	} else {
		slider.css("top", (this.config.height * this.config.position) + "px");
	}
}

Slidon.prototype.remove = function (caller, element, number) {
	if (!this.readyToRun("remove", arguments)) {
		return this;
	}

	/* If jQuery element specified, find its index */

	if (element instanceof jQuery) {
		if (!element.length) {
			return error("Slidon remove error! Invalid jQuery element, either select it properly or specify the element's index inside Slidon instead");
		}

		element = parseInt(element.attr("data-slidon-index"));

		if (isNaN(element)) { // XXX: TEST
			return error("Slidon remove error! the specified element to be removed seems not to be part of a Slidon, you either did not select it properly or your internal Slidon structure is corrupted");
		}
	}

	if (!this.config.items) {
		return error("Slidon remove error! empty Slidon, there are not elements to be removed");
	}

	if (element > this.config.items || element < 1) {
		return error("Slidon remove error! un-existent element index to remove, remember index starts at 1 and not 0, first element in the Slidon has an index of 1");
	}

	number = number || 1;
	number = ((element + number) > this.config.items) ? (this.config.items - element + 1) : number;

	/* Remove elements from Slidon */

	this.removeElements(element, number);

	/* Remove Scrolling Box elements and fix indexes */

	if (this.config.options.scroller.enabled) {
		var scroller = this.getScroller();
		var removed = scroller.children().slice(element - 1, element - 1 + number).detach();

		var self = this;
		var totalResolution = 0;

		scroller.children().each(function(index) {
			jQuery(this).attr("data-slidon-index", index + 1);
			totalResolution += (self.config.options.scroller.type == "horizontal") ? jQuery(this).outerWidth(true) : jQuery(this).outerHeight(true);
		});

		if (this.config.options.scroller.type == "horizontal") {
			scroller.width(totalResolution);
		} else {
			scroller.height(totalResolution);
		}

		if (removed.filter(".slidon-current").length) {
			var slider = this.getSlider();
			var current = slider.children(".slidon-current").attr("data-slidon-index");

			scroller.children(".slidon-current").removeClass("slidon-current");
			scroller.children("[data-slidon-index=" + current + "]").addClass("slidon-current");
		}

		removed.remove();
	}

	if (this.config.options.callbacks["removeComplete"]) {
		this.config.options.callbacks["removeComplete"](caller, element, number);
	}

	return this;
};

Slidon.prototype.queue = function (caller, action) {
	if (action == "remove") {
		this.config.queue = [];
	}

	return (action) ? this : this.config.queue;
}

Slidon.prototype.items = function(caller) {
	return this.config.items;
}

Slidon.prototype.iter = function(caller, items) {
	if (!items) {
		return this.config.options.iter;
	}

	if (!this.readyToRun("iter", arguments)) {
		return this;
	}

	items = parseInt(items);

	if (!isNaN(items) && items > 0 && items < this.config.items) {
		this.config.options.iter = items;

		if (this.config.options.callbacks["iterComplete"]) {
			this.config.options.callbacks["iterComplete"](caller, items);
		}
	}

	return this;
}

Slidon.prototype.display = function(caller, elements) {
	if (!elements) {
		return this.config.options.display;
	}

	if (elements && !this.readyToRun("display", arguments)) {
		return this;
	}

	if (elements > 0 && elements <= this.config.items) {
		this.config.options.display = elements;

		if (this.config.options.type == "horizontal") {
			this.getSlidon().width(this.config.width * elements);
		} else {
			this.getSlidon().height(this.config.height * elements);
		}

		if (this.config.options.callbacks["displayComplete"]) {
			this.config.options.callbacks["displayComplete"](caller, elements);
		}
	}

	return (elements) ? this : this.config.options.display;
}

Slidon.prototype.current = function(caller) {
	return parseInt(this.getSlidon().attr("data-slidon-current"));
}

/*
 * TODO: Let user specify direction and change Slidon type if 
 * automatically if the specified direction does not match the
 * current Slidon type. Right now you can only invert the 
 * current direction calling the invert command.
 *
 **/

Slidon.prototype.direction = function(caller) {
	return this.getSlidon().attr("data-slidon-direction");
}

Slidon.prototype.type = function(caller, type) {
	if (!type) {
		return this.config.options.type;
	}

	if (!this.readyToRun("type", arguments)) {
		return this;
	}

	if (type && this.config.options.type != type) {
		var slider = this.getSlider();
		var slidon = this.getSlidon();

		if (type == "horizontal") {
			slider.css({
				"top"		: 0,
				"right"		: (!this.config.options.carousel) ? (this.config.width * (this.current() - 1)) : 0,
			});

			slider.children().css("display", "inline-block");
			slider.width(this.config.width * this.config.items).height(this.config.height);

			slidon.width(this.config.width * this.config.options.display).height(this.config.height);
			slidon.parent().width(this.config.width * this.config.options.display).height(this.config.height);
		} else {
			slider.css({
				"right"		: 0,
				"top"		: (!this.config.options.carousel) ? (-1) * (this.config.height * (this.current() - 1)) : 0,
			});

			slider.children().css("display", "block");
			slider.width(this.config.width).height(this.config.height * this.config.items);

			slidon.width(this.config.width).height(this.config.height * this.config.options.display);
			slidon.parent().width(this.config.width).height(this.config.height * this.config.options.display);
		}

		this.config.options.type = type;

		this.config.options.direction = typeDirection(this.config.options.direction, type);
		slidon.attr("data-slidon-direction", this.config.options.direction);
	}

	return this;
};

/*
 * TODO: Enable client to set more than one button per action
 *
 **/

Slidon.prototype.button = function(caller, type, button) {
	if (!this.readyToRun("button", arguments)) {
		return this;
	}

	if (button && (!(button instanceof jQuery) || !button.length)) {
		return error("Slidon button setting error! specified '" + type + "' button is not a valid jQuery element, kindly check if you are selecting it properly. Specified button:", button);
	}

	/* Clear previous button */

	if (this.config.buttons[type]) {
		var oldButton = this.filter(jQuery("[data-slidon=" + this.config.id + "][data-slidon-button=" + type + "]"));
		oldButton.removeAttr("data-slidon-button").off("click.slidon");
	}

	/* Set button */

	this.config.buttons[type] = button;

	if (button) {
		var self = this;

		button.attr({
			"data-slidon-button": type,
			"data-slidon": this.config.id,		
		}).off("click.slidon").on("click.slidon", function() {
			self[jQuery(this).attr("data-slidon-button")]("user");
		});

		this.config.buttons[type] = true;
	}

	return this;
}

Slidon.prototype.shuffle = function (caller) {
	if (!this.readyToRun("shuffle", arguments)) {
		return this;
	}

	caller = caller || "code";
	this.config.force = false;

	var slider = this.getSlider();
	var scroller = (this.config.options.scroller.enabled) ? this.getScroller() : false;

	/* Generate new shuffle list */

	var elements = slider.children(".slidon-item");
	var shufflerOrder = shuffleArray(Array.apply(null, {length: elements.length}).map(Number.call, Number));

	/* Order elements */

	jQuery.each(shufflerOrder, function(index, slidonIndex) {
		slider.children("[data-slidon-index=" + (slidonIndex + 1) + "].slidon-item").appendTo(slider);

		if (scroller) {
			scroller.children("[data-slidon-index=" + (slidonIndex + 1) + "].slidon-scroller-item").appendTo(scroller);
		}
	});

	/* Assign new order indexes to elements */

	slider.children(".slidon-item").each(function(index, element) {
		jQuery(this).attr("data-slidon-index", index + 1);
	});

	if (scroller) {
		scroller.children(".slidon-scroller-item").each(function(index, element) {
			jQuery(this).attr("data-slidon-index", index + 1);
		});
	}

	this.config.seen = currentTime();

	/* Update "slidon-current" element */

	var position = (this.config.options.carousel) ? 0 : ((-1) * this.config.position);
	var index = slider.children(".slidon-item").removeClass("slidon-current").eq(position).addClass("slidon-current").attr("data-slidon-index");

	if (scroller) {
		scroller.children(".slidon-scroller-item.slidon-current").removeClass("slidon-current");
		scroller.children("[data-slidon-index=" + index + "].slidon-scroller-item").addClass("slidon-current");
	}

	if (this.config.options.callbacks["shuffleComplete"]) {
		this.config.options.callbacks["shuffleComplete"](caller);
	}

	return this;
}

Slidon.prototype.hide = function(caller) {
	if (!this.readyToRun("hide", arguments)) {
		return this;
	}

	caller = caller || "code";
	var scroller = this.getScroller();

	if (scroller.css("display") == "none") {
		return this;
	}

	if (this.config.options.callbacks["hideComplete"]) {
		var self = this;
		scroller.fadeOut(this.config.options.scroller.hideSpeed, function() {
			self.config.options.callbacks["hideComplete"](caller);
		});
	} else {
		scroller.fadeOut(this.config.options.scroller.hideSpeed);
	}

	return this;
}

Slidon.prototype.show = function(caller) {
	if (!this.readyToRun("show", arguments)) {
		return this;
	}

	caller = caller || "code";
	var scroller = this.getScroller();

	if (scroller.css("display") == "block") {
		return this;
	}

	if (this.config.options.callbacks["showComplete"]) {
		var self = this;
		scroller.fadeIn(this.config.options.scroller.hideSpeed, function() {
			self.config.options.callbacks["showComplete"](caller);
		});
	} else {
		scroller.fadeIn(this.config.options.scroller.hideSpeed);
	}

	return this;
}

Slidon.prototype.mouseIn = function (caller) {
	caller = caller || "code";

	if (this.config.options.callbacks["mouseIn"] && !this.config.options.callbacks["mouseIn"](caller)) {
		return false;
	}

	if (this.config.options.autoScroll.pause && this.config.options.autoScroll.handler && !this.config.options.autoScroll.bounceStop) {
		this.config.options.autoScroll.pauseStop = true;
		this.stop(caller);
	}

	return true;
}

Slidon.prototype.mouseOut = function(caller) {
	caller = caller || "code";
	if (this.config.options.callbacks["mouseOut"] && !this.config.options.callbacks["mouseOut"](caller)) {
		return false;
	}

	if (this.config.options.autoScroll.pause && this.config.options.autoScroll.pauseStop) {
		this.config.options.autoScroll.pauseStop = false;
		this.start(caller);
	}

	return true;
}

Slidon.prototype.autoScrolling = function() {
	var callbackTimeout = 100;

	/* User might had removed the Slidon from DOM, logging warning just in case */

	if (!this.getSlidon()) {
		error("Slidon warning! Stopping Auto Scroller, running over a Slidon it has already been removed from DOM");
		return false;
	}

	if (!this.config.working) {
		var callbackTimeout = this.config.options.autoScroll.time;
		var timeSeen = currentTime() - this.config.seen;

		if (timeSeen < this.config.options.autoScroll.time) {
			callbackTimeout = this.config.options.autoScroll.time - timeSeen;
		} else {
			this[this.config.options.direction]("autoScroll");
		}
	}

	var self = this;
	this.config.options.autoScroll.handler = setTimeout(function() {
		Slidon.prototype.autoScrolling.apply(self);
	}, callbackTimeout);
}

Slidon.prototype.pause = function(caller, time) {
	if (!this.readyToRun("pause", arguments)) {
		return this;
	}

	this.config.working = true;

	var self = this;
	setTimeout(function() {
		self.config.working = false;
		self.processQueue();
	}, time);
}

Slidon.prototype.stop = function(caller) {
	caller = caller || "code";

	if (this.config.options.autoScroll.enabled && this.config.options.autoScroll.handler) {
		if (this.config.options.callbacks["stop"]) {
			this.config.options.callbacks["stop"](caller);
		}

		clearTimeout(this.config.options.autoScroll.handler);
		this.config.options.autoScroll.handler = null;

		if (this.config.options.callbacks["stopComplete"]) {
			this.config.options.callbacks["stopComplete"](caller);
		}
	}

	return this;
}

Slidon.prototype.start = function(caller) {
	caller = caller || "code";
	var self = this;

	if (this.config.options.autoScroll.enabled && !this.config.options.autoScroll.handler) {
		if (this.config.options.callbacks["start"]) {
			if (!this.config.options.callbacks["start"](caller)) {
				return this;
			}
		}

		this.config.options.autoScroll.handler = setTimeout(function() {
			Slidon.prototype.autoScrolling.apply(self);
		}, 0);

		if (this.config.options.autoScroll.bounce) {
			if (((jQuery.inArray(this.config.options.direction, ["left", "up"]) > -1) && !this.config.position) ||
				((jQuery.inArray(this.config.options.direction, ["right", "down"]) > -1) && ((this.config.options.display + ((-1) * this.config.position)) == this.config.items))) {
				this.invert("autoScroll", true);
			}
		}

		if (this.config.options.callbacks["startComplete"]) {
			this.config.options.callbacks["startComplete"](caller);
		}
	}

	return this;
}

Slidon.prototype.invert = function(caller, force) {
	if (!this.readyToRun("invert", arguments)) {
		return this;
	}

	caller = caller || "code";
	var slidon = this.getSlidon();

	this.config.options.direction = invertDirection(this.config.options.direction);
	slidon.attr("data-slidon-direction", this.config.options.direction);

	if (this.config.options.callbacks["invertComplete"]) {
		this.config.options.callbacks["invertComplete"](caller);
	}

	return this;
}

Slidon.prototype.index = function(caller, index, speed) {
	if (!this.readyToRun("index", arguments)) {
		return this;
	}

	if (index instanceof jQuery) {
		if (!index.length) {
			error("Slidon index error! Invalid jQuery element, either select it properly or specify the element's index inside Slidon instead");
			return this.processQueue();
		}

		index = parseInt(element.attr("data-slidon-index"));

		if (isNaN(index)) {
			error("Slidon remove error! the specified element to be indexed seems not to be part of a Slidon, you either did not select it properly or your internal Slidon structure is corrupted");
			return this.processQueue();
		}
	}

	var slider = this.getSlider();
	var currentElement = parseInt(slider.children(".slidon-item.slidon-current").attr("data-slidon-index"));

	speed = ((speed === undefined) || isNaN(speed) || (parseInt(speed) < 0)) ? undefined : parseInt(speed);

	if ((index <= 0) || (index > this.config.items) || (index === currentElement)) {
		return this.processQueue();
	}

	this.config.force = true;	
	caller = caller || "code";

	if (currentElement < index) {
		this.right(caller, index - currentElement, "index", speed);
	} else {
		this.left(caller, currentElement - index, "index", speed);
	}

	return this;
}

Slidon.prototype.prev = function(caller, speed) {
	if (!this.readyToRun("prev", arguments)) {
		return this;
	}

	caller = caller || "code";
	this.config.force = true;

	var method = directionMethod(invertDirection(this.config.options.direction));
	this[method](caller, this.config.options.iter, "prev", speed);

	return this;
}

Slidon.prototype.next = function(caller, speed) {
	if (!this.readyToRun("next", arguments)) {
		return this;
	}

	caller = caller || "code";
	this.config.force = true;

	var method = directionMethod(this.config.options.direction);
	this[method](caller, this.config.options.iter, "next", speed);

	return this;
}

/* Linear Sliding Methods */

Slidon.prototype.moveComplete = function(caller, iter, direction, context) {
	if (!this.getSlidon()) {
		return false;
	}

	var index = ((-1) * this.config.position) + 1;
	this.setCurrentElement(index);

	if (this.config.options.callbacks[direction + "Complete"]) {
		this.config.options.callbacks[direction + "Complete"](caller);
	}

	if (context && this.config.options.callbacks[context + "Complete"]) {
		this.config.options.callbacks[context + "Complete"](caller, index);
	}

	this.config.working = false;
	this.config.seen = currentTime();

	if (((jQuery.inArray(direction, ["left", "up"]) > -1) && !this.config.position) || ((jQuery.inArray(direction, ["right", "down"]) > -1) && ((this.config.options.display + ((-1) * this.config.position)) == this.config.items))) {
		if (this.config.options.autoScroll.bounce) {
			this.invert("autoScroll", true);
		} else {
			this.config.options.autoScroll.bounceStop = true;
		}
	}

	this.processQueue();
}

Slidon.prototype.moveLeft = function(caller, iter, context, speed) {
	var direction = (this.config.options.type == "vertical") ? "up" : "left";

	if (!this.readyToRun(direction, arguments)) {
		return this;
	}

	caller = caller || "code";
	iter = iter || this.config.options.iter;

	if (!iter || (iter > this.config.items)) {
		return this.processQueue();
	}

	if (!this.config.position) {
		return this.processQueue();
	}

	if (iter > (-1 * this.config.position)) {
		iter = (-1 * this.config.position);
	}

	this.config.options.autoScroll.bounceStop = false;
	this.config.position = (this.config.position + iter);

	if (speed === undefined) {
		speed = this.processSpeed(context, iter);
	}

	this.config.working = true;
	this.config.seen = currentTime() + speed;

	var self = this;
	var slider = this.getSlider();

	var effect = (this.config.options.effect == "dynamic") ? this.randomEffect() : this.config.options.effect;
	this.config.lastEffect = effect;

	if (this.config.options.type === "horizontal") {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var right = parseInt((slider.css("right") == "auto") ? 0 : slider.css("right")) - (self.config.width * iter);

				slider.css({'right': right}).fadeIn(self.config.fadeSpeed, function(){
					self.moveComplete(caller, iter, "left", context);
				});
			});

		} else {
			slider.animate({'right': '-=' + (this.config.width * iter) + 'px'}, speed, effect, function() {
				self.moveComplete(caller, iter, "left", context);
			});
		}

	} else {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var top = parseInt((slider.css("top") == "auto") ? 0 : slider.css("top")) + (self.config.height * iter);

				slider.css({'top': top}).fadeIn(self.config.fadeSpeed, function(){
					self.moveComplete(caller, iter, "up", context);
				});
			});

		} else {
			slider.animate({'top': '+=' + (this.config.height * iter) + 'px'}, speed, effect, function() {
				self.moveComplete(caller, iter, "up", context)
			});
		}
	}	

	this.processScrollerMove("left", iter-1, speed/iter, context);
	return this;
}

Slidon.prototype.moveRight = function(caller, iter, context, speed) {
	var direction = (this.config.options.type == "vertical") ? "down" : "right";

	if (!this.readyToRun(direction, arguments)) {
		return this;
	}

	caller = caller || "code";
	iter = iter || this.config.options.iter;

	if (((this.config.items - this.config.options.display) + this.config.position) < iter) {
		iter = (this.config.items - this.config.options.display) + this.config.position;
	}

	if (!iter || (iter > this.config.items)) {
		return this.processQueue();
	}

	if ((this.config.items - this.config.options.display) == this.config.position) {
		return this.processQueue();
	}

	this.config.options.autoScroll.bounceStop = false;
	this.config.position = this.config.position - iter;

	if (speed === undefined) {
		speed = this.processSpeed(context, iter);
	}

	var self = this;
	var slider = this.getSlider();

	var effect = (this.config.options.effect == "dynamic") ? this.randomEffect() : this.config.options.effect;
	this.config.lastEffect = effect;

	this.config.working = 1;
	this.config.seen = currentTime() + speed;

	if (this.config.options.type == "horizontal") {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var right = parseInt((slider.css("right") == "auto") ? 0 : slider.css("right")) + (self.config.width * iter);

				slider.css({'right': right}).fadeIn(self.config.fadeSpeed, function(){
					self.moveComplete(caller, iter, "right", context);
				});
			});

		} else {
			slider.animate({'right': '+=' + (this.config.width * iter) + 'px'}, speed, effect, function() {
				self.moveComplete(caller, iter, "right", context);
			});
		}

	} else {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var top = parseInt((slider.css("top") == "auto") ? 0 : slider.css("top")) - (self.config.height * iter);

				slider.css({'top': top}).fadeIn(self.config.fadeSpeed, function(){
					self.moveComplete(caller, iter, "down", context);
				});
			});

		} else {
			slider.animate({'top': '-=' + (this.config.height * iter) + 'px'}, speed, effect, function() {
				self.moveComplete(caller, iter, "down", context);
			});
		}
	}

	this.processScrollerMove("right", iter-1, speed/iter, context);
	return this;
}

/* Carousel Sliding Functions */

Slidon.prototype.moveCompleteCarousel = function(caller, iter, direction, context) {
	if (!this.getSlidon()) {
		return false;
	}

	var slider = this.getSlider();
	this.config.seen = currentTime();

	if (jQuery.inArray(direction, ["right", "down"]) != -1) {
		slider.children(".slidon-item").slice(0, iter).remove();
	} else {
		slider.children(".slidon-item").slice((-1) * iter).remove();
	}

	if (this.config.options.type == "horizontal") {
		slider.width(slider.width() - (this.config.width * iter)).css("right", 0);
	} else {
		slider.height(slider.height() - (this.config.height * iter)).css("top", 0);
	}

	/* Set Slidon current element */

	var index = slider.children(".slidon-item:first").attr("data-slidon-index");
	this.setCurrentElement(index);

	if (this.config.options.callbacks[direction + "Complete"]) {
		this.config.options.callbacks[direction + "Complete"](caller);
	}

	if (context && this.config.options.callbacks[context + "Complete"]) {
		this.config.options.callbacks[context + "Complete"](caller, index);
	}

	this.config.working = false;
	this.processQueue();
}

Slidon.prototype.moveRightCarousel = function (caller, iter, context, speed) {
	var direction = (this.config.options.type == "vertical") ? "down" : "right";

	if (!this.readyToRun(direction, arguments)) {
		return this;
	}

	caller = caller || "code";
	iter = iter || this.config.options.iter;

	if (!iter || (iter > this.config.items)) {
		return this.processQueue();
	}

	var slider = this.getSlider();

	if (this.config.options.type == "horizontal") {
		slider.width(slider.width() + (this.config.width * iter));
	} else {
		slider.height(slider.height() + (this.config.height * iter));
	}

	slider.append(slider.children(".slidon-item").slice(0, iter).addClass("slidon-remove").clone(true, true).removeClass("slidon-remove"));

	if (speed === undefined) {
		speed = this.processSpeed(context, iter);
	}

	var self = this;

	var effect = (this.config.options.effect == "dynamic") ? this.randomEffect() : this.config.options.effect;
	this.config.lastEffect = effect;

	this.config.working = true;
	this.config.seen = currentTime() + speed;

	if (this.config.options.type == "horizontal") {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var right = parseInt((slider.css("right") == "auto") ? 0 : slider.css("right")) + (self.config.width * iter);

				slider.css({'right': right}).fadeIn(self.config.fadeSpeed, function(){
					self.moveCompleteCarousel(caller, iter, "right", context);
				});
			});

		} else {
			slider.animate({'right': '+=' + (this.config.width * iter) + 'px'}, speed, effect, function() {
				self.moveCompleteCarousel(caller, iter, "right", context);
			});
		}
	} else {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var top = parseInt((slider.css("top") == "auto") ? 0 : slider.css("top")) - (self.config.height * iter);

				slider.css({'top': top}).fadeIn(self.config.fadeSpeed, function(){
					self.moveCompleteCarousel(caller, iter, "down", context);
				});
			});
		} else {
			slider.animate({'top': '-=' + (this.config.height * iter) + 'px'}, speed, effect, function() {
				self.moveCompleteCarousel(caller, iter, "down", context);
			});
		}
	}

	this.processScrollerMove("right", iter-1, speed/iter, context);
	return this;
}

Slidon.prototype.moveLeftCarousel = function (caller, iter, context, speed) {
	var direction = (this.config.options.type == "vertical") ? "up" : "left";

	if (!this.readyToRun(direction, arguments)) {
		return this;
	}

	caller = caller || "code";
	iter = iter || this.config.options.iter;

	if (!iter || (iter > this.config.items)) {
		return this.processQueue();
	}

	var slider = this.getSlider();

	if (this.config.options.type == "horizontal") {
		slider.width(slider.width() + (this.config.width * iter));

		var right = (slider.css("right") != "auto") ? ("+=" + (this.config.width * iter) + "px") : ((this.config.width * iter) + "px");
		slider.prepend(slider.children(".slidon-item").slice((-1) * iter).addClass("slidon-remove").clone(true, true).removeClass("slidon-remove")).css("right", right);

	} else {
		slider.height(slider.height() + (this.config.height * iter));

		var top = (slider.css("top") != "auto") ? ("-=" + (this.config.height * iter) + "px") : (((-1) * this.config.height * iter) + "px");
		slider.prepend(slider.children(".slidon-item").slice((-1) * iter).addClass("slidon-remove").clone(true, true).removeClass("slidon-remove")).css("top", top);
	}

	if (speed === undefined) {
		speed = this.processSpeed(context, iter);
	}

	var self = this;

	var effect = (this.config.options.effect == "dynamic") ? this.randomEffect() : this.config.options.effect;
	this.config.lastEffect = effect;

	this.config.working = true;
	this.config.seen = currentTime() + speed;

	if (this.config.options.type == "horizontal") {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var right = parseInt((slider.css("right") == "auto") ? 0 : slider.css("right")) - (self.config.width * iter);

				slider.css({'right': right}).fadeIn(self.config.fadeSpeed, function(){
					self.moveCompleteCarousel(caller, iter, "left", context);
				});
			});

		} else {
			slider.animate({'right': '-=' + (this.config.width * iter) + 'px'}, speed, effect, function() {
				self.moveCompleteCarousel(caller, iter, "left", context)
			});
		}

	} else {
		if (effect == "fade") {
			this.config.fadeSpeed = speed/2;
			slider.fadeOut(this.config.fadeSpeed, function() {
				var top = parseInt((slider.css("top") == "auto") ? 0 : slider.css("top")) + (self.config.height * iter);

				slider.css({'top': top}).fadeIn(self.config.fadeSpeed, function(){
					self.moveCompleteCarousel(caller, iter, "up", context);
				});
			});
		} else {
			slider.animate({'top': '+=' + (this.config.height * iter) + 'px'}, speed, effect, function() {
				self.moveCompleteCarousel(caller, iter, "up", context)
			});
		}
	}

	this.processScrollerMove("left", iter-1, speed/iter, context);
	return this;
}

Slidon.prototype.processScrollerMove = function(direction, iter, speed, context) {
	if (iter && speed && ((context != "index") || !this.config.options.instantIndex)) {
		var self = this;
		setTimeout(function() {
			self.autoScrollMove(iter, speed, direction);
		}, speed);
	}
}

/*
 * Scroller "slidon-current" class animation.
 *
 * NOTE: For some easing effects, due to the way they increase and decrease their
 * speed at different points of the easing effect, when scrolling using some 
 * effects the scrolling move of slidon-current class might not follow properly
 * the easing effect as i'm calculating intervals periods as if time over elements
 * was linear.
 * 
 * TODO: In next version i will see the way of getting each easing effect
 * time periods and movement pattern and calculate the intervals perfectly.
 *
 **/

Slidon.prototype.autoScrollMove = function(iter, speed, direction) {
	var slider = this.getSlider();
	var scroller = this.getScroller();
	var currentIndex = parseInt(slider.children(".slidon-item.slidon-current").attr("data-slidon-index"));

	if ((currentIndex == 1) && (jQuery.inArray(direction, ["up", "left"]) != -1)) {
		var index = this.config.items;
	} else if ((currentIndex == this.config.items) && (jQuery.inArray(direction, ["right", "down"]) != -1)) {
		var index = 1;
	} else if (jQuery.inArray(direction, ["right", "down"]) != -1) {
		var index = currentIndex + 1;
	} else {
		var index = currentIndex - 1;
	}

	slider.children(".slidon-item.slidon-current").removeClass("slidon-current");
	slider.children("[data-slidon-index=" + index + "].slidon-item").addClass("slidon-current");

	if (scroller.length) {
		scroller.children(".slidon-scroller-item.slidon-current").removeClass("slidon-current");
		scroller.children("[data-slidon-index=" + index + "].slidon-scroller-item").addClass("slidon-current");
	}

	if (--iter) {
		var self = this;
		setTimeout(function() {
			self.autoScrollMove(iter, speed, direction);
		}, speed);
	}
}

Slidon.prototype.command = function(method) {
	var commands = [
		"next", 
		"prev",
		"up",
		"down",
		"right",
		"left",
		"invert",
		"shuffle",
		"index",
		"stop",
		"hide",
		"show",
		"start",
		"speed",
		"effect",
		"button",
		"callback",
		"pause",
		"type",
		"current",
		"direction",
		"items",
		"queue",
		"add",
		"remove"
	];

	if (!this.getSlidon()) {
		error("Slidon command error! You are trying to send a command to a Slidon it has been removed from DOM");
		return false;
	}

	if (!this[method]) {
		return error("Slidon command error! '" + method + "' command not supported. Supported commands: " + commands.join(", "));
	}

	var args = Array.apply({}, arguments);
	args.shift();
	args.unshift("code");

	return this[method].apply(this, args);
}

Slidon.prototype.effect = function(caller, effect) {
	if (!this.readyToRun("effect", arguments)) {
		return this;
	}

	if (jQuery.inArray(effect, this.easingEffects.concat(["random", "dynamic", "fade"])) == -1) {
		return error("Slidon effect setting error! '" + effect + "' effect is not supported, supported effects: random, dynamic, fade, " + this.easingEffects.join(", "));
	}

	if (effect == "random") {
		effect = this.randomEffect();
	}

	this.config.options.effect = effect;

	if (this.config.options.callbacks["effectComplete"]) {
		this.config.options.callbacks["effectComplete"](caller, effect);
	}

	return this;
}

Slidon.prototype.speed = function(caller, speed) {
	if (!this.readyToRun("speed", arguments)) {
		return this;
	}

	var speed = parseInt(speed);

	if (speed instanceof Array) {
		if (isNaN(speed[0]) || (speed[0] < 0) || isNaN(speed[1]) || (speed[1] < 0) || (speed[1] >= speed[0])) {
			return error("Slidon speed setting error! minimum/maximum speed must be >= 0 and minimum speed must be < than maximum speed");
		}
	} else {
		if (isNaN(speed) || (speed < 0)) {
			return error("Slidon speed setting error! speed must be >= 0");
		}
	}

	this.config.options.speed = speed;
	
	if (this.config.options.callbacks["speedComplete"]) { 
		this.config.options.callbacks["speedComplete"](caller, speed);
	}

	return this;
}

Slidon.prototype.processQueue = function() {
	while (this.config.queue.length && !this.config.working) {
		var call = this.config.queue.shift();

		this.config.force = true;
		this[call[0]].apply(this, call[1]);

		/* 
		 * The following actions finish in a time in the near future, we stop
		 * executing any further actions in queue as when this actions are
		 * completed they will call this functions to keep processing queue.
		 *
		 **/

		if (jQuery.inArray(call[0], ["index", "up", "down", "right", "left", "next", "prev", "pause"]) != -1) {
			break;
		}
	}

	return this;
};

Slidon.prototype.callback = function(caller, callback, func) {
	if (!this.readyToRun("callback", arguments)) {
		return this;
	}

	caller = caller || "code";
	this.config.options.callbacks[callback] = func;

	if (this.config.options.callbacks["callbackComplete"]) {
		this.config.options.callbacks["callbackComplete"](caller, callback, func);
	}

	return this;
};

/* 
 * TODO: Check if minifying worked setting the default values like
 * this and if not have separate default objects and extend options.
 *
 **/

Slidon.prototype.validateOptions = function(options) {
	options = options || { autoScroll: { time: 1000 } };

	var validPositions = [
		"top left",
		"top center",
		"top right",
		"middle left",
		"middle center",
		"middle right",
		"bottom left",
		"bottom center",
		"bottom right",
	];

	/* Lets validate some parameters */

	if (("type" in options) && options["type"] != "horizontal" && options["type"] != "vertical") {
		throw new Error("Slidon settings error! invalid 'type' value. Valid options: horizontal, vertical");
	}

	if (("direction" in options) && jQuery.inArray(options["direction"], ["up", "down", "right", "left"]) == -1) {
		throw new Error("Slidon settings error! invalid 'direction' value. Valid options: 'up', 'down', 'left', 'right'");
	}

	if (("iter" in options) && (parseInt(options["iter"]) <= 0)) {
		throw new Error("Slidon settings error! invalid 'iter' value. Must be >= 1");
	}

	if (("display" in options) && (parseInt(options["display"]) <= 0)) {
		throw new Error("Slidon settings error! invalid 'display' value. Must be >= 1");
	}

	if ("scroller" in options) {
		if (("proportion" in options["scroller"]) && (options["scroller"]["proportion"] < 1 || options["scroller"]["proportion"] > 9)) {
			throw new Error("Slidon settings error! invalid 'proportion' value. Must be >= 1 and <= 9");
		}

		if (("position" in options["scroller"]) && (jQuery.inArray(options["scroller"]["position"], validPositions) == -1)) {
			throw new Error("Slidon settings error! invalid 'position' value. Valid values: " + validPositions.join(", "));
		}
	}

	if (options["effect"] && jQuery.inArray(options["effect"], this.easingEffects.concat(["random", "dynamic","fade"])) == -1) {
		throw new Error("Slidon settings error! specified effect is not supported, supported effects: random, dynamic, fade" + this.easingEffects.join(", "));
	}

	/* Check callbacks are really callbacks :P */

	if ("callbacks" in options) {
		jQuery.each(options["callbacks"], function(key, value) {
			if (value && !jQuery.isFunction(value)) {
				throw new Error("Slidon settings error! specified '" + key + "' callback is not a function");
			}
		});
	} else {
		options["callbacks"] = {};
	}

	var settings = {};
	settings.buttons = options["buttons"] || null;

	/* Process main default settings */

	settings.iter = options["iter"] || 1;
	settings.display = options["display"] || 1;
	settings.speed = (options["speed"] === undefined || (options["speed"] < 0)) ? 1000 : options["speed"];

	if (settings.speed instanceof Array && (settings.speed.length != 2)) {
		throw new Error("Slidon settings error! when setting minimum/maximum speed, parameter must be an array with two elements: [minSpeed, maxSpeed]");
	}

	settings.type = options["type"] || "horizontal";
	settings.carousel = (options["carousel"] !== undefined) ? options["carousel"] : true;
	settings.direction = options["direction"] || (options["type"] == "horizontal") ? "right" : "down";

	settings.index = options["index"] || 1;
	settings.instantIndex = options["instantIndex"] || false;

	settings.debug = options["debug"] || false;
	settings.effect = options["effect"] || "dynamic";
	settings.forceStyling = options["forceStyling"] || false;

	if (settings.effect == "random") {
		settings.effect = this.randomEffect();
	}

	/* Process Auto Scroller default options */

	settings.autoScroll = {};
	settings.autoScroll.enabled = ("autoScroll" in options) ? true : false;

	if (!settings.autoScroll.enabled) {
		options["autoScroll"] = {};
	}

	settings.autoScroll.time = options["autoScroll"]["time"] || 2000;
	settings.autoScroll.pause = options["autoScroll"]["pause"] || false;
	settings.autoScroll.bounce = options["autoScroll"]["bounce"] || false;

	/* Process Scrolling Box default options */

	settings.scroller = {};
	settings.scroller.enabled = ("scroller" in options) ? true : false;

	if (settings.scroller.enabled) {
		settings.scroller.type = options["scroller"]["type"] || "horizontal";
		settings.scroller.position = options["scroller"]["position"] || "bottom center";

		settings.scroller.hidden = options["scroller"]["hidden"] || false;
		settings.scroller.hideSpeed = options["scroller"]["hideSpeed"] || 300;
		settings.scroller.proportion = options["scroller"]["proportion"] || false;
	}

	/* Process callbacks default values */

	settings.callbacks = {};
	
	if(!("callbacks" in options)) {
		options["callbacks"] = {};
	}

	var callbacks = ["next", "prev", "right", "left", "up", "down", "start", "stop", "show", "hide", "index", "shuffle", "invert", "speed", "effect"];

	jQuery.each(callbacks, function(index, callback) {
		settings.callbacks[callback] = options["callbacks"][callback] || null;
		settings.callbacks[callback + "Complete"] = options["callbacks"][callback + "Complete"] || null;
	})

	settings.callbacks["ready"] = options["callbacks"]["ready"] || null;
	settings.callbacks["mouseIn"] = options["callbacks"]["mouseIn"] || null;
	settings.callbacks["mouseOut"] = options["callbacks"]["mouseOut"] || null;

	return settings;
}

/*
 * Finally! If you are here I guess you have been checking the Slidon code, 
 * first of all, I hope it was not a pain in the ass reading it! second and 
 * even more important, if there are any questions you might want to ask me 
 * or even request any feature you might need that Slidon still doesn't have 
 * and would really make your developing more easy feel free to send me an 
 * email and I will try to help you out or add the feature for you.
 *
 */

module.exports = Slidon;
