/**
 * Feature: Bootstrap 5 renderer layout options and configurations
 * Background: fixture = /packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html
 */

/*
 * CHECKLIST — Scenarios in this spec
 * [x] creates horizontal layout by default
 * [x] creates vertical layout when specified
 * [x] applies vertical button classes correctly
 * [x] handles vertical button text overrides
 * [x] switches between horizontal and vertical layouts
 * [x] maintains Bootstrap grid compatibility
 * [x] supports responsive behavior
 * [x] handles size variants (sm, lg) in both layouts
 * [x] applies proper spacing in vertical layout
 * [x] manages button positioning in vertical mode
 * [x] handles prefix/postfix in vertical layout
 * [x] handles vertical buttons with postfix only (no prefix)
 * [x] maintains accessibility in both layouts
 * [x] supports custom vertical button classes
 * [x] handles layout changes after initialization
 * [x] preserves Bootstrap 5 component structure in both modes
 * [x] applies proper flexbox classes for layout
 * [x] handles edge cases with container constraints
 * [x] maintains proper tab order in both layouts
 * [x] supports nested layout scenarios
 * [x] handles dynamic content changes in layouts
 */

import { expect, test } from '@playwright/test';
import * as apiHelpers from '@touchspin/core/test-helpers';
import { bootstrap5RendererUrl, ensureBootstrap5Globals } from './helpers/bootstrap5-globals';

/**
 * Scenario: creates horizontal layout by default
 * Given the fixture page is loaded
 * When TouchSpin initializes with Bootstrap5 renderer
 * Then buttons are arranged horizontally around the input
 * Params:
 * { "defaultLayout": "horizontal", "expectedStructure": "down-input-up", "layoutDirection": "row" }
 */
test('creates horizontal layout by default', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify horizontal layout structure
  await expect(elements.wrapper).toHaveClass(/input-group/);
  await expect(elements.upButton).toBeVisible();
  await expect(elements.downButton).toBeVisible();

  // Verify functionality works
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: creates vertical layout when specified
 * Given the fixture page is loaded
 * When TouchSpin initializes with verticalbuttons option
 * Then buttons are arranged vertically beside the input
 * Params:
 * { "verticalbuttons": true, "expectedStructure": "input-with-vertical-stack", "layoutDirection": "column" }
 */
test('creates vertical layout when specified', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify vertical layout structure
  await expect(elements.wrapper).toHaveClass(/input-group/);
  await expect(elements.upButton).toBeVisible();
  await expect(elements.downButton).toBeVisible();

  // Verify functionality works in vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: applies vertical button classes correctly
 * Given the fixture page is loaded
 * When TouchSpin initializes with vertical layout and custom classes
 * Then buttons have both base and vertical-specific classes
 * Params:
 * { "verticalupclass": "btn-up-vertical", "verticaldownclass": "btn-down-vertical", "expectedMerging": "base_plus_vertical" }
 */
test('applies vertical button classes correctly', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
    verticalupclass: 'btn btn-success v-up',
    verticaldownclass: 'btn btn-warning v-down',
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify vertical-specific classes are applied
  await expect(elements.upButton).toHaveClass(/btn-success/);
  await expect(elements.downButton).toHaveClass(/btn-warning/);

  // Verify functionality works
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: handles vertical button text overrides
 * Given the fixture page is loaded
 * When TouchSpin initializes with vertical-specific button texts
 * Then vertical buttons display the specified text
 * Params:
 * { "verticalup": "▲", "verticaldown": "▼", "expectedDisplay": "vertical_arrows" }
 */
test('handles vertical button text overrides', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
    verticalup: '▲',
    verticaldown: '▼',
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify vertical-specific text is displayed
  await expect(elements.upButton).toHaveText('▲');
  await expect(elements.downButton).toHaveText('▼');

  // Verify functionality works
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: switches between horizontal and vertical layouts
 * Given the fixture page is loaded with initialized TouchSpin
 * When the layout is changed via settings update
 * Then the DOM rebuilds with the new layout
 * Params:
 * { "initialLayout": "horizontal", "newLayout": "vertical", "expectedBehavior": "rebuild_layout" }
 */
test('switches between horizontal and vertical layouts', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  // Start with horizontal layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Switch to vertical layout
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  // Verify functionality still works after layout switch
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');

  // Switch back to horizontal
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: false,
  });

  await apiHelpers.clickDownButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: maintains Bootstrap grid compatibility
 * Given the fixture page is loaded within Bootstrap grid system
 * When TouchSpin initializes with either layout
 * Then it works correctly within grid constraints
 * Params:
 * { "gridContext": "col-md-6", "layoutTypes": ["horizontal", "vertical"], "expectedCompatibility": true }
 */
test('maintains Bootstrap grid compatibility', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/layout-options-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'grid-test', bootstrap5RendererUrl);

  // Test both horizontal and vertical layouts in grid context
  await apiHelpers.clickUpButton(page, 'grid-test');
  await apiHelpers.expectValueToBe(page, 'grid-test', '51');

  // Switch to vertical and test
  await apiHelpers.updateSettingsViaAPI(page, 'grid-test', {
    verticalbuttons: true,
  });

  await apiHelpers.clickUpButton(page, 'grid-test');
  await apiHelpers.expectValueToBe(page, 'grid-test', '52');
});

/**
 * Scenario: supports responsive behavior
 * Given the fixture page is loaded
 * When viewport size changes with TouchSpin in different layouts
 * Then the component responds appropriately to size changes
 * Params:
 * { "viewportSizes": ["mobile", "tablet", "desktop"], "expectedBehavior": "responsive_adaptation" }
 */
test('supports responsive behavior', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  // Test at mobile viewport
  await page.setViewportSize({ width: 375, height: 667 });
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Test at tablet viewport
  await page.setViewportSize({ width: 768, height: 1024 });
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');

  // Test at desktop viewport
  await page.setViewportSize({ width: 1200, height: 800 });
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '53');
});

/**
 * Scenario: handles size variants (sm, lg) in both layouts
 * Given the fixture page is loaded with sized inputs
 * When TouchSpin initializes with different layouts
 * Then size variants work correctly in both horizontal and vertical modes
 * Params:
 * { "sizeVariants": ["sm", "normal", "lg"], "layoutTypes": ["horizontal", "vertical"], "expectedBehavior": "size_preserved" }
 */
test('handles size variants (sm, lg) in both layouts', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/layout-options-fixture.html');
  await ensureBootstrap5Globals(page);

  // Test actual pre-existing size variant inputs (should work with fix)
  // This will expose the bug if the renderer fix is not working properly

  // Test small size input in input-group-sm
  await apiHelpers.initializeTouchSpin(page, 'size-sm-test', bootstrap5RendererUrl);

  // Verify input is still accessible and functional
  await apiHelpers.expectValueToBe(page, 'size-sm-test', '50');
  await apiHelpers.clickUpButton(page, 'size-sm-test');
  await apiHelpers.expectValueToBe(page, 'size-sm-test', '51');

  // Test large size input in input-group-lg
  await apiHelpers.initializeTouchSpin(page, 'size-lg-test', bootstrap5RendererUrl);

  // Verify input is still accessible and functional
  await apiHelpers.expectValueToBe(page, 'size-lg-test', '50');
  await apiHelpers.clickUpButton(page, 'size-lg-test');
  await apiHelpers.expectValueToBe(page, 'size-lg-test', '51');

  // Test layout switch on sized inputs (this was the critical bug trigger)
  await apiHelpers.updateSettingsViaAPI(page, 'size-sm-test', { verticalbuttons: true });
  await apiHelpers.clickUpButton(page, 'size-sm-test');
  await apiHelpers.expectValueToBe(page, 'size-sm-test', '52');

  // Switch back to horizontal (this was where the DOM error occurred)
  await apiHelpers.updateSettingsViaAPI(page, 'size-sm-test', { verticalbuttons: false });

  // Verify it still works after layout switch back to horizontal
  await apiHelpers.clickUpButton(page, 'size-sm-test');
  await apiHelpers.expectValueToBe(page, 'size-sm-test', '53');

  // Test the same cycle with large input
  await apiHelpers.updateSettingsViaAPI(page, 'size-lg-test', { verticalbuttons: true });
  await apiHelpers.clickUpButton(page, 'size-lg-test');
  await apiHelpers.expectValueToBe(page, 'size-lg-test', '52');

  // Switch large input back to horizontal
  await apiHelpers.updateSettingsViaAPI(page, 'size-lg-test', { verticalbuttons: false });
  await apiHelpers.clickUpButton(page, 'size-lg-test');
  await apiHelpers.expectValueToBe(page, 'size-lg-test', '53');
});

/**
 * Scenario: applies proper spacing in vertical layout
 * Given the fixture page is loaded
 * When TouchSpin initializes with vertical layout
 * Then proper spacing is applied between stacked buttons
 * Params:
 * { "verticalbuttons": true, "expectedSpacing": "bootstrap_button_group_spacing", "spacingClasses": ["btn-group-vertical"] }
 */
test('applies proper spacing in vertical layout', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify buttons are visible and functional (spacing is handled by Bootstrap)
  await expect(elements.upButton).toBeVisible();
  await expect(elements.downButton).toBeVisible();

  // Test that spacing doesn't interfere with functionality
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  await apiHelpers.clickDownButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '50');
});

/**
 * Scenario: manages button positioning in vertical mode
 * Given the fixture page is loaded
 * When TouchSpin initializes with vertical layout
 * Then buttons are positioned correctly relative to input
 * Params:
 * { "verticalbuttons": true, "expectedPositioning": "beside_input", "buttonOrder": ["up", "down"] }
 */
test('manages button positioning in vertical mode', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify both buttons are positioned correctly and clickable
  await expect(elements.upButton).toBeVisible();
  await expect(elements.downButton).toBeVisible();

  // Test that positioning allows proper interaction
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  await apiHelpers.clickDownButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '50');
});

/**
 * Scenario: handles prefix/postfix in vertical layout
 * Given the fixture page is loaded
 * When TouchSpin initializes with vertical layout and prefix/postfix
 * Then prefix and postfix are positioned correctly with vertical buttons
 * Params:
 * { "verticalbuttons": true, "prefix": "$", "postfix": "USD", "expectedLayout": "prefix-input-postfix-vertical-buttons" }
 */
test('handles prefix/postfix in vertical layout', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
    prefix: '$',
    postfix: 'USD',
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify prefix and postfix are present with vertical layout
  await expect(elements.prefix).toBeVisible();
  await expect(elements.prefix).toHaveText('$');
  await expect(elements.postfix).toBeVisible();
  await expect(elements.postfix).toHaveText('USD');

  // Verify functionality works with prefix/postfix in vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: handles vertical buttons with postfix only (no prefix)
 * Given the fixture page is loaded
 * When TouchSpin initializes with vertical layout and postfix but no prefix
 * Then input is positioned correctly before postfix, with vertical buttons after
 * Params:
 * { "verticalbuttons": true, "postfix": "USD", "expectedLayout": "input-postfix-vertical-buttons" }
 */
test('handles vertical buttons with postfix only (no prefix)', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
    postfix: 'USD',
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify postfix is present and positioned correctly
  await expect(elements.postfix).toBeVisible();
  await expect(elements.postfix).toHaveText('USD');

  // Verify vertical buttons are visible
  await expect(elements.upButton).toBeVisible();
  await expect(elements.downButton).toBeVisible();

  // Verify functionality works with postfix in vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');
});

/**
 * Scenario: maintains accessibility in both layouts
 * Given the fixture page is loaded
 * When TouchSpin initializes with either layout
 * Then accessibility attributes and tab order are correct
 * Params:
 * { "layoutTypes": ["horizontal", "vertical"], "accessibilityChecks": ["aria-labels", "tab-order", "keyboard-navigation"] }
 */
test('maintains accessibility in both layouts', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);

  // Test horizontal layout accessibility
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify accessibility attributes are present
  await expect(elements.upButton).toHaveAttribute('type', 'button');
  await expect(elements.downButton).toHaveAttribute('type', 'button');

  // Test functionality in horizontal layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Switch to vertical layout and verify accessibility is maintained
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  const verticalElements = await apiHelpers.getTouchSpinElements(page, 'test-input');
  await expect(verticalElements.upButton).toHaveAttribute('type', 'button');
  await expect(verticalElements.downButton).toHaveAttribute('type', 'button');

  // Test functionality in vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');
});

/**
 * Scenario: supports custom vertical button classes
 * Given the fixture page is loaded
 * When TouchSpin initializes with custom vertical button classes
 * Then the custom classes are applied in addition to defaults
 * Params:
 * { "verticalupclass": "custom-up", "verticaldownclass": "custom-down", "expectedBehavior": "additive_classes" }
 */
test('supports custom vertical button classes', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    verticalbuttons: true,
    verticalupclass: 'btn btn-success custom-up',
    verticaldownclass: 'btn btn-danger custom-down',
  });

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify custom classes are applied
  await expect(elements.upButton).toHaveClass(/btn-success/);
  await expect(elements.upButton).toHaveClass(/custom-up/);
  await expect(elements.downButton).toHaveClass(/btn-danger/);
  await expect(elements.downButton).toHaveClass(/custom-down/);

  // Verify functionality works with custom classes
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  await apiHelpers.clickDownButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '50');
});

/**
 * Scenario: handles layout changes after initialization
 * Given the fixture page is loaded with initialized TouchSpin
 * When layout is changed dynamically via API
 * Then the layout updates without losing state
 * Params:
 * { "initialValue": "50", "layoutChange": "horizontal_to_vertical", "expectedBehavior": "preserve_state" }
 */
test('handles layout changes after initialization', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', {
    initval: 75,
  });

  // Verify initial value and horizontal layout functionality
  await apiHelpers.expectValueToBe(page, 'test-input', '75');
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '76');

  // Change to vertical layout and verify state is preserved
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  // State should be preserved after layout change
  await apiHelpers.expectValueToBe(page, 'test-input', '76');

  // Verify functionality works in new layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '77');

  // Switch back to horizontal layout
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: false,
  });

  // State should still be preserved
  await apiHelpers.expectValueToBe(page, 'test-input', '77');
  await apiHelpers.clickDownButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '76');
});

/**
 * Scenario: preserves Bootstrap 5 component structure in both modes
 * Given the fixture page is loaded
 * When TouchSpin initializes with different layouts
 * Then Bootstrap 5 component integrity is maintained
 * Params:
 * { "layoutTypes": ["horizontal", "vertical"], "componentIntegrity": "bootstrap5_standards", "expectedCompliance": true }
 */
test('preserves Bootstrap 5 component structure in both modes', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);

  // Test horizontal layout Bootstrap structure
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify Bootstrap 5 structure for horizontal layout
  await expect(elements.wrapper).toHaveClass(/input-group/);
  await expect(elements.upButton).toHaveClass(/btn/);
  await expect(elements.downButton).toHaveClass(/btn/);

  // Test functionality in horizontal layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Switch to vertical layout and verify Bootstrap structure is maintained
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  const verticalElements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify Bootstrap 5 structure for vertical layout
  await expect(verticalElements.wrapper).toHaveClass(/input-group/);
  await expect(verticalElements.upButton).toHaveClass(/btn/);
  await expect(verticalElements.downButton).toHaveClass(/btn/);

  // Test functionality in vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');
});

/**
 * Scenario: applies proper flexbox classes for layout
 * Given the fixture page is loaded
 * When TouchSpin initializes with different layouts
 * Then appropriate flexbox classes are applied for each layout
 * Params:
 * { "horizontalClasses": ["d-flex"], "verticalClasses": ["flex-column"], "expectedBehavior": "layout_specific_classes" }
 */
test('applies proper flexbox classes for layout', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);

  // Test horizontal layout flexbox classes
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify horizontal layout has appropriate structure (Bootstrap handles flexbox internally)
  await expect(elements.wrapper).toHaveClass(/input-group/);

  // Test functionality in horizontal layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Switch to vertical layout and verify appropriate structure
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  const verticalElements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify vertical layout maintains Bootstrap structure
  await expect(verticalElements.wrapper).toHaveClass(/input-group/);

  // Test functionality in vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');
});

/**
 * Scenario: handles edge cases with container constraints
 * Given the fixture page is loaded with constrained containers
 * When TouchSpin initializes with different layouts
 * Then it adapts gracefully to container limitations
 * Params:
 * { "containerConstraints": ["narrow_width", "limited_height"], "expectedBehavior": "graceful_adaptation" }
 */
test('handles edge cases with container constraints', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/layout-options-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'constrained-test', bootstrap5RendererUrl);

  // Verify functionality works in constrained container
  await apiHelpers.clickUpButton(page, 'constrained-test');
  await apiHelpers.expectValueToBe(page, 'constrained-test', '51');

  // Test vertical layout in constrained container
  await apiHelpers.updateSettingsViaAPI(page, 'constrained-test', {
    verticalbuttons: true,
  });

  // Verify functionality still works with vertical layout in constraints
  await apiHelpers.clickUpButton(page, 'constrained-test');
  await apiHelpers.expectValueToBe(page, 'constrained-test', '52');

  await apiHelpers.clickDownButton(page, 'constrained-test');
  await apiHelpers.expectValueToBe(page, 'constrained-test', '51');
});

/**
 * Scenario: maintains proper tab order in both layouts
 * Given the fixture page is loaded
 * When TouchSpin initializes with different layouts
 * Then keyboard navigation follows logical tab order
 * Params:
 * { "layoutTypes": ["horizontal", "vertical"], "expectedTabOrder": ["down-button", "input", "up-button"], "keyboardNavigation": true }
 */
test('maintains proper tab order in both layouts', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);

  // Test horizontal layout tab order
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  const elements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify elements are focusable (tab order is maintained by Bootstrap structure)
  await expect(elements.downButton).toBeVisible();
  await expect(elements.input).toBeVisible();
  await expect(elements.upButton).toBeVisible();

  // Test functionality with horizontal layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Switch to vertical layout and verify tab order is maintained
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  const verticalElements = await apiHelpers.getTouchSpinElements(page, 'test-input');

  // Verify elements remain focusable in vertical layout
  await expect(verticalElements.downButton).toBeVisible();
  await expect(verticalElements.input).toBeVisible();
  await expect(verticalElements.upButton).toBeVisible();

  // Test functionality with vertical layout
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');
});

/**
 * Scenario: supports nested layout scenarios
 * Given the fixture page is loaded with nested containers
 * When TouchSpin initializes within nested layouts
 * Then it works correctly in complex layout scenarios
 * Params:
 * { "nestingScenarios": ["card_within_card", "modal_content"], "expectedBehavior": "nested_compatibility" }
 */
test('supports nested layout scenarios', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/layout-options-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'nested-test', bootstrap5RendererUrl);

  // Verify functionality works in nested layout
  await apiHelpers.clickUpButton(page, 'nested-test');
  await apiHelpers.expectValueToBe(page, 'nested-test', '51');

  // Test vertical layout in nested scenario
  await apiHelpers.updateSettingsViaAPI(page, 'nested-test', {
    verticalbuttons: true,
  });

  // Verify functionality still works with vertical layout in nested structure
  await apiHelpers.clickUpButton(page, 'nested-test');
  await apiHelpers.expectValueToBe(page, 'nested-test', '52');

  await apiHelpers.clickDownButton(page, 'nested-test');
  await apiHelpers.expectValueToBe(page, 'nested-test', '51');
});

/**
 * Scenario: handles dynamic content changes in layouts
 * Given the fixture page is loaded with initialized TouchSpin
 * When content around the component changes dynamically
 * Then the layout remains stable and functional
 * Params:
 * { "dynamicChanges": ["sibling_elements_added", "parent_resized"], "expectedBehavior": "layout_stability" }
 */
test('handles dynamic content changes in layouts', async ({ page }) => {
  await page.goto('/packages/renderers/bootstrap5/tests/fixtures/bootstrap5-fixture.html');
  await ensureBootstrap5Globals(page);
  await apiHelpers.initializeTouchSpin(page, 'test-input', bootstrap5RendererUrl);

  // Test initial functionality
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '51');

  // Add dynamic content around the component
  await page.evaluate(() => {
    const wrapper = document.querySelector('[data-testid="test-input"]').closest('.mb-4');

    // Add sibling elements
    const siblingBefore = document.createElement('div');
    siblingBefore.className = 'alert alert-info';
    siblingBefore.textContent = 'Dynamic content before';
    wrapper.parentNode.insertBefore(siblingBefore, wrapper);

    const siblingAfter = document.createElement('div');
    siblingAfter.className = 'alert alert-warning';
    siblingAfter.textContent = 'Dynamic content after';
    wrapper.parentNode.insertBefore(siblingAfter, wrapper.nextSibling);

    // Resize parent container
    wrapper.parentNode.style.width = '80%';
  });

  // Verify functionality remains stable after dynamic changes
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');

  // Switch to vertical layout and test with dynamic content
  await apiHelpers.updateSettingsViaAPI(page, 'test-input', {
    verticalbuttons: true,
  });

  // Verify functionality works with vertical layout and dynamic content
  await apiHelpers.clickUpButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '53');

  await apiHelpers.clickDownButton(page, 'test-input');
  await apiHelpers.expectValueToBe(page, 'test-input', '52');
});
