/* InterActiveMultiSelect jQuery Plugin v1.0.1
   - Dropdown-only multi-select
   - Checkbox or radio mode
   - Select All / Clear links
   - Row highlighting for selected items
   - Search/filter option
   - Keyboard navigation support
   - ARIA accessibility
   - Keeps original <select> in sync for form submission (ASP.NET MVC compatible)
   - Bootstrap & Bootstrap modal compatible
*/
(function($){
  'use strict';

  var defaults = {
    mode: 'checkbox',
    placeholder: 'Select',
    search: false,
    searchPlaceholder: 'Search...',
    noResultsText: 'No results found',
    selectAllText: 'Select All',
    clearText: 'Clear'
  };

  var KEYS = {
    ENTER: 13,
    ESC: 27,
    SPACE: 32,
    UP: 38,
    DOWN: 40,
    TAB: 9
  };

  function build($select, opts){
    var name = $select.attr('name') || '';
    var uid = 'ias_' + Math.random().toString(36).slice(2);

    // Create wrapper and button with ARIA
    var $wrapper = $('<div class="ias-wrapper"></div>');
    var $button = $('<div class="ias-btn" tabindex="0" role="combobox" aria-haspopup="listbox" aria-expanded="false" aria-controls="' + uid + '_list"></div>');
    var $tagsContainer = $('<div class="ias-tags" aria-live="polite"></div>');
    var $placeholder = $('<span class="ias-placeholder">' + opts.placeholder + '</span>');
    var $arrow = $('<span class="ias-arrow" aria-hidden="true">&#9662;</span>');
    $button.append($tagsContainer).append($placeholder).append($arrow);
    
    // Create dropdown with ARIA
    var $dropdown = $('<div class="ias-dropdown" role="dialog" aria-label="Options"></div>');

    // Controls: Select All / Clear
    var $controls = $('<div class="ias-controls"></div>');
    var $selectAll = $('<a href="#" class="ias-select-all" role="button">' + opts.selectAllText + '</a>');
    var $clear = $('<a href="#" class="ias-clear" role="button">' + opts.clearText + '</a>');
    
    if (opts.mode === 'checkbox') {
      $controls.append($selectAll).append($clear);
    } else {
      $controls.append($clear);
    }
    $dropdown.append($controls);

    // Search input (optional)
    var $searchWrap = null;
    var $searchInput = null;
    var $noResults = $('<div class="ias-no-results" role="status">' + opts.noResultsText + '</div>');
    
    if (opts.search) {
      $searchWrap = $('<div class="ias-search"></div>');
      $searchInput = $('<input type="text" class="ias-search-input" placeholder="' + opts.searchPlaceholder + '" aria-label="Search options" />');
      $searchWrap.append($searchInput);
      $dropdown.append($searchWrap);

      // Search filter handler
      $searchInput.on('input', function(){
        var query = $(this).val().toLowerCase().trim();
        var visibleCount = 0;
        
        // When search is cleared, show all items and headers immediately
        if (query === '') {
          $list.find('.ias-item').show();
          $list.find('.ias-group-header').show();
          $noResults.hide();
          return;
        }
        
        $list.find('.ias-item').each(function(){
          var $item = $(this);
          var text = $item.find('label').text().toLowerCase();
          if (text.indexOf(query) > -1) {
            $item.show();
            visibleCount++;
          } else {
            $item.hide();
          }
        });
        
        // Hide group headers if no visible items in group
        $list.find('.ias-group-header').each(function(){
          var $header = $(this);
          var $nextItems = $header.nextUntil('.ias-group-header', '.ias-item:visible');
          if ($nextItems.length === 0) {
            $header.hide();
          } else {
            $header.show();
          }
        });
        
        // Show/hide no results message
        if (visibleCount === 0) {
          $noResults.show();
        } else {
          $noResults.hide();
        }
      });

      // Keyboard in search
      $searchInput.on('keydown', function(e){
        if (e.which === KEYS.DOWN) {
          e.preventDefault();
          focusNextItem($list, null);
        } else if (e.which === KEYS.ESC) {
          closeDropdown($wrapper, $button);
          $button.focus();
        }
      });

      $searchInput.on('click', function(e){
        e.stopPropagation();
      });
    }

    // List container with ARIA
    var $list = $('<div class="ias-list" id="' + uid + '_list" role="listbox" aria-multiselectable="' + (opts.mode === 'checkbox' ? 'true' : 'false') + '"></div>');

    // Build items from original select options (with optgroup support)
    var itemIndex = 0;
    
    $select.children('optgroup, option').each(function(){
      var $this = $(this);
      
      // Handle optgroup
      if ($this.is('optgroup')) {
        var groupLabel = $this.attr('label') || '';
        var $groupHeader = $('<div class="ias-group-header"></div>').text(groupLabel);
        $list.append($groupHeader);
        
        // Process options inside optgroup
        $this.find('option').each(function(){
          var $opt = $(this);
          buildItem($opt, itemIndex++);
        });
      } else {
        // Handle standalone option
        var $opt = $this;
        var val = $opt.attr('value');
        if (val === '' || val === undefined) return; // Skip empty options
        buildItem($opt, itemIndex++);
      }
    });
    
    function buildItem($opt, i){
      var val = $opt.attr('value');
      var text = $opt.text();
      var disabled = $opt.is(':disabled') || $opt.parent('optgroup').is(':disabled');
      
      if (val === '' || val === undefined) return; // Skip empty options

      var inputType = (opts.mode === 'radio') ? 'radio' : 'checkbox';
      var inputName = (inputType === 'radio') ? uid : null;

      var $item = $('<div class="ias-item" role="option" tabindex="-1" aria-selected="false"></div>');
      var id = uid + '_' + i;
      var $input = $('<input type="' + inputType + '" id="' + id + '" value="' + val + '" tabindex="-1" />');
      if (inputName) $input.attr('name', inputName);
      if (disabled) {
        $input.prop('disabled', true);
        $item.attr('aria-disabled', 'true');
      }
      
      var $label = $('<label for="' + id + '"></label>').text(text);

      if ($opt.is(':selected')) {
        $input.prop('checked', true);
        $item.addClass('selected').attr('aria-selected', 'true');
      }

      $item.append($input).append($label);
      $item.data('value', val);
      if (disabled) $item.addClass('disabled');
      $list.append($item);

      // Click handler
      $item.on('click', function(e){
        e.stopPropagation();
        if (disabled) return;
        toggleItem($item, $input, $select, $list, $button, $wrapper, opts, val, inputType);
      });

      // Keyboard handler for items
      $item.on('keydown', function(e){
        handleItemKeyboard(e, $item, $input, $select, $list, $button, $wrapper, opts, val, inputType, $searchInput);
      });
    }

    // Append no results message (hidden by default)
    $noResults.hide();
    $list.append($noResults);

    $dropdown.append($list);

    // Attach to wrapper
    $wrapper.append($button).append($dropdown);

    // Hide original select, insert wrapper after it
    $select.hide().after($wrapper);

    // Store reference for destroy
    $select.data('ias-wrapper', $wrapper);

    // Toggle dropdown on button click
    $button.on('click', function(e){
      e.stopPropagation();
      toggleDropdown($wrapper, $button, $searchInput);
    });

    // Button keyboard handler
    $button.on('keydown', function(e){
      if (e.which === KEYS.ENTER || e.which === KEYS.SPACE) {
        e.preventDefault();
        toggleDropdown($wrapper, $button, $searchInput);
      } else if (e.which === KEYS.DOWN) {
        e.preventDefault();
        if (!$wrapper.hasClass('open')) {
          openDropdown($wrapper, $button, $searchInput);
        }
        if (opts.search && $searchInput) {
          $searchInput.focus();
        } else {
          focusNextItem($list, null);
        }
      } else if (e.which === KEYS.ESC) {
        closeDropdown($wrapper, $button);
      }
    });

    // Close dropdown when clicking outside
    $(document).on('click.ias.' + uid, function(e){
      if (!$(e.target).closest('.ias-wrapper').length) {
        closeDropdown($wrapper, $button);
      }
    });

    // Close on ESC globally
    $(document).on('keydown.ias.' + uid, function(e){
      if (e.which === KEYS.ESC && $wrapper.hasClass('open')) {
        closeDropdown($wrapper, $button);
        $button.focus();
      }
    });

    // Prevent dropdown from closing when clicking inside
    $dropdown.on('click', function(e){
      e.stopPropagation();
    });

    // Select All handler
    $selectAll.on('click', function(e){
      e.preventDefault();
      $list.find('.ias-item:visible:not(.disabled)').each(function(){
        var $item = $(this);
        var $input = $item.find('input');
        $input.prop('checked', true);
        $item.addClass('selected').attr('aria-selected', 'true');
        var val = $input.val();
        $select.find('option[value="' + escapeVal(val) + '"]').prop('selected', true);
      });
      updateButtonLabel($button, $select, opts, $list);
      $select.trigger('change');
    });

    // Clear handler
    $clear.on('click', function(e){
      e.preventDefault();
      $list.find('.ias-item:visible:not(.disabled)').each(function(){
        var $item = $(this);
        var $input = $item.find('input');
        $input.prop('checked', false);
        $item.removeClass('selected').attr('aria-selected', 'false');
        var val = $input.val();
        $select.find('option[value="' + escapeVal(val) + '"]').prop('selected', false);
      });
      updateButtonLabel($button, $select, opts, $list);
      $select.trigger('change');
    });

    // Store uid for cleanup
    $select.data('ias-uid', uid);

    // Initial button label
    updateButtonLabel($button, $select, opts, $list);
  }

  function toggleItem($item, $input, $select, $list, $button, $wrapper, opts, val, inputType){
    if (inputType === 'checkbox') {
      var checked = !$input.prop('checked');
      $input.prop('checked', checked);
      $item.toggleClass('selected', checked).attr('aria-selected', checked ? 'true' : 'false');
      $select.find('option[value="' + escapeVal(val) + '"]').prop('selected', checked);
    } else {
      // Radio: uncheck others
      $list.find('input[type="radio"]').prop('checked', false);
      $list.find('.ias-item').removeClass('selected').attr('aria-selected', 'false');
      $input.prop('checked', true);
      $item.addClass('selected').attr('aria-selected', 'true');
      $select.find('option').prop('selected', false);
      $select.find('option[value="' + escapeVal(val) + '"]').prop('selected', true);
      // Close dropdown after radio selection
      closeDropdown($wrapper, $button);
    }
    updateButtonLabel($button, $select, opts, $list);
    $select.trigger('change');
  }

  function handleItemKeyboard(e, $item, $input, $select, $list, $button, $wrapper, opts, val, inputType, $searchInput){
    switch(e.which){
      case KEYS.ENTER:
      case KEYS.SPACE:
        e.preventDefault();
        if (!$item.hasClass('disabled')) {
          toggleItem($item, $input, $select, $list, $button, $wrapper, opts, val, inputType);
        }
        break;
      case KEYS.DOWN:
        e.preventDefault();
        focusNextItem($list, $item);
        break;
      case KEYS.UP:
        e.preventDefault();
        focusPrevItem($list, $item, $searchInput);
        break;
      case KEYS.ESC:
        closeDropdown($wrapper, $button);
        $button.focus();
        break;
      case KEYS.TAB:
        closeDropdown($wrapper, $button);
        break;
    }
  }

  function focusNextItem($list, $current){
    var $items = $list.find('.ias-item:visible:not(.disabled)');
    if ($items.length === 0) return;
    
    if (!$current) {
      $items.first().focus();
    } else {
      var idx = $items.index($current);
      if (idx < $items.length - 1) {
        $items.eq(idx + 1).focus();
      }
    }
  }

  function focusPrevItem($list, $current, $searchInput){
    var $items = $list.find('.ias-item:visible:not(.disabled)');
    if ($items.length === 0) return;
    
    var idx = $items.index($current);
    if (idx > 0) {
      $items.eq(idx - 1).focus();
    } else if ($searchInput) {
      $searchInput.focus();
    }
  }

  function toggleDropdown($wrapper, $button, $searchInput){
    if ($wrapper.hasClass('open')) {
      closeDropdown($wrapper, $button);
    } else {
      openDropdown($wrapper, $button, $searchInput);
    }
  }

  function openDropdown($wrapper, $button, $searchInput){
    // Close all other dropdowns
    $('.ias-wrapper.open').each(function(){
      var $w = $(this);
      $w.removeClass('open');
      $w.find('.ias-btn').attr('aria-expanded', 'false');
    });
    $wrapper.addClass('open');
    $button.attr('aria-expanded', 'true');
    
    // Focus search if available
    if ($searchInput) {
      setTimeout(function(){ $searchInput.focus(); }, 50);
    }
  }

  function closeDropdown($wrapper, $button){
    $wrapper.removeClass('open');
    $button.attr('aria-expanded', 'false');
    
    // Clear search
    var $searchInput = $wrapper.find('.ias-search-input');
    if ($searchInput.length) {
      $searchInput.val('').trigger('input');
    }
  }

  function updateButtonLabel($button, $select, opts, $list){
    var selected = $select.find('option:selected').filter(function(){
      return $(this).val() !== '';
    });
    var $tagsContainer = $button.find('.ias-tags');
    var $placeholder = $button.find('.ias-placeholder');
    
    // Clear existing tags
    $tagsContainer.empty();
    
    if (selected.length === 0) {
      $placeholder.show();
    } else {
      $placeholder.hide();
      selected.each(function(){
        var $opt = $(this);
        var val = $opt.attr('value');
        var text = $opt.text();
        var $tag = $('<span class="ias-tag"></span>');
        var $tagText = $('<span class="ias-tag-text"></span>').text(text);
        var $tagRemove = $('<span class="ias-tag-remove" role="button" aria-label="Remove ' + text + '" tabindex="0" data-value="' + escapeVal(val) + '">&times;</span>');
        $tag.append($tagText).append($tagRemove);
        $tagsContainer.append($tag);
        
        // Remove tag click handler
        $tagRemove.on('click keydown', function(e){
          if (e.type === 'keydown' && e.which !== KEYS.ENTER && e.which !== KEYS.SPACE) return;
          e.preventDefault();
          e.stopPropagation();
          var removeVal = $(this).data('value');
          // Uncheck in list
          if ($list) {
            $list.find('input[value="' + escapeVal(removeVal) + '"]').prop('checked', false)
              .closest('.ias-item').removeClass('selected').attr('aria-selected', 'false');
          }
          // Deselect in original select
          $select.find('option[value="' + escapeVal(removeVal) + '"]').prop('selected', false);
          updateButtonLabel($button, $select, opts, $list);
          $select.trigger('change');
        });
      });
    }
  }

  function escapeVal(str){
    return String(str).replace(/["'\\]/g, '\\$&');
  }

  function destroy($select){
    var uid = $select.data('ias-uid');
    var $wrapper = $select.data('ias-wrapper');
    
    if ($wrapper) {
      // Remove event listeners
      $(document).off('.ias.' + uid);
      $wrapper.remove();
    }
    
    // Show original select
    $select.show();
    $select.removeData('ias-init');
    $select.removeData('ias-uid');
    $select.removeData('ias-wrapper');
  }

  function refresh($select){
    var opts = $select.data('ias-opts') || defaults;
    destroy($select);
    $select.interActiveMultiSelect(opts);
  }

  $.fn.interActiveMultiSelect = function(options){
    // Handle method calls
    if (typeof options === 'string') {
      var args = Array.prototype.slice.call(arguments, 1);
      return this.each(function(){
        var $this = $(this);
        if (options === 'destroy') {
          destroy($this);
        } else if (options === 'refresh') {
          refresh($this);
        } else if (options === 'val') {
          // Get or set value
          if (args.length > 0) {
            // Set value
            $this.val(args[0]).trigger('change');
            refresh($this);
          }
        }
      });
    }

    var opts = $.extend({}, defaults, options);
    return this.each(function(){
      var $this = $(this);
      if ($this.data('ias-init')) return; // Prevent double init
      $this.data('ias-init', true);
      $this.data('ias-opts', opts);
      build($this, opts);
    });
  };

})(jQuery);
