/**
 * @license
 * @summary     QbSlider
 * @description Simple slider
 * @version     0.1.3
 * @file        jquery.qbslider.js
 * @author      Autioch
 * @contact     qb.net.pl
 *
 * @copyright Copyright 2014 Autioch, all rights reserved.
 */

/*
 * The MIT License (MIT)
 * Copyright (c) 2014 Autioch
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. 
 */
;
(function (root, factory) {

    'use strict';

    if (typeof define === 'function' && define.amd) {

        define(['jquery'], factory);

    } else {

        if (jQuery && !jQuery.fn.qbslider) {

            factory(root.jQuery);

        }
    }

}(this, function ($) {


'use strict';

var defaults = {
    /* Index of slide to show after initialization */
    start: 0,
    /* Animation speed */
    speed: 500,
    /* Number of slides that will be visible at once, current slide is always centered */
    slidesVisible: 1,
    /* Autoresizing of slides after window resizing */
    responsive: true,
    /* Generate navigation links */
    nav: true,
    /* Text of the navigation links - receives index and slide element */
    navText: function (i, el) {
        return (i + 1);
    },
    mousewheel: true,
    /* Generate next and previous slide buttons */
    controls: true,
    /* Text of the next slide button */
    next: '&#9679;',
    /* Text of the previous slide button */
    prev: '&#9679;'
};

/* helper function for finding index depending on argument passed */
var _findIndexValue = function (arg) {

    /* just return a number */
    if (typeof arg === 'number') {
        return arg;
    }

    /* event passed with data attached */
    if (arg.data) {

        /*data as a single value */
        if (typeof arg.data === 'number') {
            return arg.data;
        }

        /* data as an object containing index value */
        if (arg.data.index) {
            return parseInt(arg.data.index, 10);
        }

    }

    /* if we couldn't find our index value, return false */
    return false;
};

/* Basic css that must be applied to the $el to make slider work */
var _baseCss = {
    position: 'absolute',
    padding: 0,
    margin: 0,
    top: 0,
    left: 0,
    display: 'table'
};

/* private handler for mousewheel event - requires binding to the slider object */
var _handleMouseWheel = function (ev) {

    ev.preventDefault();
    ev.cancelBubble = false;

    var change = (ev.type == 'mousewheel') ? ev.originalEvent.wheelDelta : ev.originalEvent.detail * -1;

    if (change > 0) {
        this.prevSlide();
    } else {
        this.nextSlide();
    }

    return false;
};

var transistionSupport = function () {
    var s = document.createElement('p').style;
    return 'transition' in s || 'WebkitTransition' in s || 'MozTransition' in s || 'msTransition' in s || 'OTransition' in s;
};

var cssChangeAnimate = function ($el, margin, speed) {
    $el.stop(true).animate({'margin-left': margin}, speed);
};

var cssChangeTransition = function ($el, margin) {
    $el.css('margin-left', margin);
};

var cssChange = transistionSupport() ? cssChangeTransition : cssChangeAnimate;

/* Binding methods directly in the constructor allows easier access to those methods later, 
 * constrasting to defining methods in prototype, which is cool, but we don't really need it here. */
var QbSlider = function ($el, options) {

    /* Configuration merged with user defined options */
    var config = $.extend(true, {}, defaults, options);

    /* Object for all private elements. Contains active slide index
     * and all cached dom elements, they have to be nullified to get GC */
    var controls = {
        /* wrapper around the whole slider stuff */
        wrap: $('<div class="qbs-wrap" />'),
        /* container for the slider */
        slider: $('<div class="qbs-slider" />'),
        /* previous slider button */
        prev: $('<span class="qbs-button qbs-prev" />'),
        /* next slider button */
        next: $('<span class="qbs-button qbs-next" />'),
        /* navigation container */
        nav: $('<div class="qbs-nav" />'),
        /* just slides */
        slides: $el.children(),
        /* active slide */
        active: $el.children(':nth-child(' + config.start + ')'),
        /* Index of the active slide */
        index: config.start,
        /* the initial list */
        el: $el
    };

    /* Exposed access to the currently active slide element */
    this.active = function () {
        return controls.active;
    };

    /* Exposed index of the active slide */
    this.index = function () {
        return controls.index;
    }.bind(this);

    /* Exposed function to set next slide */
    this.nextSlide = function () {
        if ((controls.index + 1) < controls.slides.length) {
            this.setSlide(controls.index + 1);
        }
    }.bind(this);

    /* Exposed function to set previous slide */
    this.prevSlide = function () {
        if (controls.index > 0) {
            this.setSlide(controls.index - 1);
        }
    }.bind(this);

    /* Exposed function to set any slide */
    this.setSlide = function (index) {

        var newindex = _findIndexValue(index);

        if (typeof newindex !== 'number' || newindex < 0 || controls.slides.length < newindex) {
            return;
        }

        /* setting active slide */
        controls.active.removeClass('qbs-active');
        controls.active = controls.slides.eq(newindex).addClass('qbs-active');

        /* index for the new active slide */
        controls.index = newindex;

        /* if navigation is set, set active link */
        if (config.nav) {
            controls.nav.children().removeClass('qbs-link-active').eq(newindex).addClass('qbs-link-active');
        }

        /* animating the change - this function is generated based on the transistion support */
        cssChange(controls.el, (controls.slider.width() - controls.active.width()) / 2 - controls.active.position().left, config.speed);
        /*controls.el.stop(true).animate({
         'margin-left': (controls.slider.width() - controls.active.width()) / 2 - controls.active.position().left
         }, config.speed);*/

        /* return false for easier event attachment for links */
        return false;

    }.bind(this);

    this.addSlide = function (slide) {

        var content = config.navText(controls.el.children().length, slide);

        if (config.nav) {
            controls.nav.append('<span class="qbs-link" data-index="' + controls.el.children().length + '">' + content + '</span>');
        }

        var size = controls.slider.width() / config.slidesVisible;

        slide.appendTo(controls.el).wrapInner('<div class="qbs-slide-content" />').css({width: size, display: 'table-cell'});

        controls.el.css('width', size * controls.el.children().length);
        controls.slides = controls.el.children();
    }.bind(this);

    /* Recalculate slider if needed */
    this.resizeSlides = function () {
        var size = controls.slider.width() / config.slidesVisible;
        controls.el.children().css('width', size);
        controls.el.css('width', size * controls.el.children().length);
        controls.slider.css('height', controls.el.height());
        this.setSlide(controls.index);
    }.bind(this);

    /* Exposed cleanup function */
    this.destroy = function () {

        var i;

        /* stop animation */
        controls.el.stop(true);

        /* unbind events */
        if (config.controls) {
            controls.prev.off('click');
            controls.next.off('click');
        }
        if (config.nav) {
            controls.nav.off('click', '.qbs-link');
        }
        if (config.responsive) {
            $(window).off('resize', this.resizeSlides);
        }

        /* remove created structure */
        controls.wrap.before(controls.el).remove();

        /* cleanup slides */
        controls.slides.removeAttr('style');
        $('.qbs-slide-content').children().unwrap();
        controls.active.removeClass('qbs-active');

        /* cleanup slide list */
        controls.el.removeData('qbslider').removeAttr('style');
        controls.el = null;

        /* cleanup all references to the removed elements and all properties of this object */
        for (i in controls) {
            controls[i] = null;
        }

        for (i in this) {
            this[i] = null;
        }

        controls = null;
        config = null;


    }.bind(this);

    /* Initialize the slider */
    var init = function () {

        /* Build slider */
        controls.el.after(controls.wrap);
        controls.slider.append(controls.el).appendTo(controls.wrap);
        controls.slides.wrapInner('<div class="qbs-slide-content" />');

        /* Add controls */
        if (config.controls) {
            controls.prev.html(config.prev).appendTo(controls.wrap).on('click', this.prevSlide);
            controls.next.html(config.next).appendTo(controls.wrap).on('click', this.nextSlide);
        }

        /* Add navigation */
        if (config.nav) {
            controls.nav.appendTo(controls.wrap);
            controls.slides.each(function (i, $el) {
                controls.nav.append('<span class="qbs-link" data-index="' + i + '">' + config.navText(i, $el) + '</span>');
            });
            controls.nav.on('click.qbs', '.qbs-link', function () {
                controls.el.data('qbslider').setSlide(parseInt($(this).attr('data-index'), 10));
            });
        }

        /* binding mousewheel events */
        if (config.mousewheel) {
            controls.wrap.on('mousewheel.qbs DOMMouseScroll.qbs', _handleMouseWheel.bind(this));
        }

        /* Setting the most important styles, without them slider will not work */
        /* Slide list container */
        controls.slider.css({
            overflow: 'hidden',
            position: 'relative'
        });

        /* If animation is transition, add add required styling */
        if (transistionSupport()) {
            controls.el.css('transition', 'margin-left ' + config.speed + 'ms ' + controls.el.css('transition'));
        }

        /* Add more required css and assign slider object to controls.el. */
        controls.el.css(_baseCss).data('qbslider', this).children().css('display', 'table-cell');

        this.resizeSlides();

        if (config.responsive) {
            $(window).on('resize', this.resizeSlides);
        }
    }.bind(this);

    init();
};

 
    // Create the jQuery plugin
    $.fn.qbslider = function (options, param) {

        var qbs;

        /* this plugin only uses  a single param for setSlide
         * so there's no need to slice args, just use second param */
        //var params = Array.prototype.slice.call(arguments, 1);

        /* return this for chainability */
        return this.each(function () {
            /* get object attached to list */
            qbs = $(this).data('qbslider');

            /* if qbs isn't our slider, initialize slider*/
            if (!(qbs instanceof QbSlider)) {
                return new QbSlider($(this), options);
            }

            /* if it is already initialized and has requested method */
            if (qbs[options] && (typeof qbs[options] === 'function')) {

                /* call the method with arguments except the first one (method name) */
                qbs[options].call(qbs, param);
            }

        });
    };

    // Expose defaults (for overriding dedault settings for all sliders)
    $.fn.qbslider.defaults = defaults;
}));