/// <reference path="./jquery.topzindex.1.2/jquery.topzindex.js" />

// desktopify-js [experimental]
// A jQuery plugin for creating a simple desktop environment with drag & drop support
// Last modified: 2020/05/07 02:17:42
// by Midas Wouters

"use strict";

(function ($) {

	$.desktopify = function (container, options) {

		// public vars
		// make reference to "options"
		this.settings = options;

		// private vars

		const _prototype = $.desktopify.prototype;
		const self = this;

		let dragStarted = false, dragging = false, selecting = false, hovering = false;
		let canDropItems = false, canDropData = false, reposition = false;

		// mouse coordinates
		let x1 = 0, y1 = 0, x2 = 0, y2 = 0;

		let closestLeft, closestRight, currentIndex;

		const styles = { position: 'absolute', display: 'none' };

		const indicatorHtml = options.indicatorHtml != '' ? options.indicatorHtml : '<div/>'; // dummy indicator

		const $container = $(container);
		const $element = (options.element != '' ? $(options.element) : $container).attr('tabindex', 0);
		const $rectangle = $(options.rectangleHtml).css(styles).appendTo($element);
		const $indicator = $(indicatorHtml).css(styles).appendTo($container);

		const element = $element[0];

		let $ghostContainer = $([]);

		let containerOffset;

		let scrollOffsetStartX, scrollDistanceX;
		let scrollOffsetStartY, scrollDistanceY;

		let positions = [];
		let centers = [];
		let $icons = $([]);

		// private methods

		function resetScrollDistance() {
			scrollOffsetStartX = container.scrollLeft;
			scrollOffsetStartY = container.scrollTop;
			scrollDistanceX = 0;
			scrollDistanceY = 0;
		}

		function updateScrollDistance() {
			scrollDistanceX = scrollOffsetStartX - container.scrollLeft;
			scrollDistanceY = scrollOffsetStartY - container.scrollTop;
		}

		function calculateRect() {

			const extraX = element == container ? container.scrollLeft : 0;
			const extraY = element == container ? container.scrollTop : 0;

			const j1 = x1 + extraX + scrollDistanceX;
			const j2 = x2 + extraX;
			const k1 = y1 + extraY + scrollDistanceY;
			const k2 = y2 + extraY;

			const offs1 = $element.offset();

			offs1.left += parseInt($element.css('border-left-width')) || 0;
			offs1.top += parseInt($element.css('border-top-width')) || 0;

			let x3 = Math.min(j1, j2) - offs1.left;
			let y3 = Math.min(k1, k2) - offs1.top;
			const x4 = Math.max(j1, j2) - offs1.left;
			const y4 = Math.max(k1, k2) - offs1.top;

			// calculate bounds
			const b_x3 = Math.max(x3, 0);
			const b_y3 = Math.max(y3, 0);
			const b_x4 = Math.min(x4, Math.max($element.width(), element.scrollWidth - element.scrollLeft + extraX));
			const b_y4 = Math.min(y4, Math.max($element.height(), element.scrollHeight - element.scrollTop + extraY));

			$rectangle.css({
				left: b_x3 + 'px',
				top: b_y3 + 'px',
				width: b_x4 - b_x3 - (parseInt($rectangle.css('border-left-width')) || 0) * 2 + 'px',
				height: b_y4 - b_y3 - (parseInt($rectangle.css('border-top-width')) || 0) * 2 + 'px',
			});

			const offs2 = {
				left: containerOffset.left + (parseInt($container.css('border-left-width')) || 0) + extraX - offs1.left - container.scrollLeft,
				top: containerOffset.top + (parseInt($container.css('border-top-width')) || 0) + extraY - offs1.top - container.scrollTop
			};

			$icons.each(function (i, el) {

				const center = {
					x: centers[i].left + offs2.left,
					y: centers[i].top + offs2.top
				};

				if (center.y > y3 && center.y < y4 && center.x > x3 && center.x < x4)
					$(el).addClass(options.selectingClass);
				else
					$(el).removeClass(options.selectingClass);
			});
		}

		function destroyGhosts() {
			$ghostContainer.remove();
		}

		function createGhosts() {
			$ghostContainer = $('<div/>').addClass(options.ghostContainerClass).appendTo($('body'));

			let minX = null, maxX = null, minY = null, maxY = null;

			const $items = $(_prototype.dragData);

			$items.each(function (i, el) {
				const $el = $(el);
				const pos = $el.position();

				minX = minX != null ? Math.min(minX, pos.left) : pos.left;
				maxX = maxX != null ? Math.max(maxX, pos.left + $el.outerWidth(true)) : pos.left + $el.outerWidth(true);
				minY = minY != null ? Math.min(minY, pos.top) : pos.top;
				maxY = maxY != null ? Math.max(maxY, pos.top + $el.outerHeight(true)) : pos.top + $el.outerHeight(true);
			});

			$items.each(function (i, el) {
				const $el = $(el);
				const pos = $el.position();

				$el.clone()
					.removeClass(options.selectedClass)
					.addClass(options.ghostClass)
					.css({
						'left': pos.left - minX + 'px',
						'top': pos.top - minY + 'px',
						'position': 'absolute',
						'cursor': 'default'
					})
					.appendTo($ghostContainer);
			});

			$ghostContainer.css({
				'pointer-events': 'none',
				'z-index': $.topZIndex(),
				'position': 'absolute',
				'overflow': 'hidden',
				'display': 'none'
			})

			if (_prototype.supportsPointerEvents) {
				const offs = {
					left: containerOffset.left + (parseInt($container.css('border-left-width')) || 0),
					top: containerOffset.top + (parseInt($container.css('border-top-width')) || 0)
				};

				$ghostContainer.data('bounds', {
					left: minX + offs.left,
					top: minY + offs.top,
					right: maxX + offs.left,
					bottom: maxY + offs.top
				});
			}
			else {
				$ghostContainer.data('bounds', {
					left: x1 + options.ghostOffset,
					top: y1 + options.ghostOffset,
					right: x1 + options.ghostOffset + (maxX - minX),
					bottom: y1 + options.ghostOffset + (maxY - minY)
				});
			}
		}

		// public methods

		this.update = function (full) {

			if (typeof full === 'undefined') full = true;

			$indicator.hide();

			const $icons_prev = $icons;
			$icons = $container.find('.' + options.iconClass);

			if (!full) {
				full = $icons.length != $icons_prev.length;

				if (!full) {
					for (let i = 0, n = $icons.length; i < n; ++i) {
						if ($icons[i] !== $icons_prev[i]) {
							full = true;
							break;
						}
					}
				}
			}

			if (full) {

				$icons.each(function (i, el) {
					const $el = $(el);
					const pos = $el.position();

					positions[i] = {
						top: pos.top + container.scrollTop,
						left: pos.left + container.scrollLeft
					};

					centers[i] = {
						top: positions[i].top + $el.outerHeight(true) / 2,
						left: positions[i].left + $el.outerWidth(true) / 2
					};
				});

				// update container offset
				containerOffset = $container.offset();
			}
		}

		this.init = function () {

			self.update();

			resetScrollDistance();

			$container.scroll(function (e) {
				if (selecting) {
					updateScrollDistance();
					calculateRect();
				}
			});

			$(window).resize(function () {
				self.update();
			});

			if (options.exclude != '') {
				$element.on('mousedown touchstart', options.exclude, function (e) {
					e.stopPropagation();
				});
			}

			$container.on('mouseenter', '.' + options.iconClass, function (e) {
				$icons.removeClass(options.dropTargetClass);

				canDropItems = _prototype.dragData.length != 0
					&& $.inArray(this, _prototype.dragData) == -1
					&& options.onBeforeHover(this, _prototype.dragData) == true;

				if (_prototype.dragging && canDropItems) {
					$(this).addClass(options.dropTargetClass);
					hovering = true;
					options.onHover(this, _prototype.dragData);
				}
			});

			$container.on('mouseleave', '.' + options.iconClass, function (e) {
				hovering = false;
				if (_prototype.supportsPointerEvents) {
					$(this).removeClass(options.dropTargetClass);
				}
			});

			$container.on('mouseup', '.' + options.iconClass, function (e) {
				$icons.removeClass(options.dropTargetClass);

				if ($.inArray(this, _prototype.dragData) != -1)
					canDropItems = false;

				if (_prototype.dragging && canDropItems) {
					options.onDropItems(this, _prototype.dragData);
					_prototype.dragData = [];
					self.update();
				}
			});

			$container.on('mousedown touchstart', '.' + options.iconClass, function (e) {

				self.update();

				x1 = x2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageX : e.pageX;
				y1 = y2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageY : e.pageY;

				if (!$(e.currentTarget).hasClass(options.selectedClass)) { // unselected item was clicked
					if (!e.ctrlKey)
						$icons.removeClass(options.selectedClass);
					$(e.currentTarget).addClass(options.selectedClass);
				}
				else if (e.ctrlKey) {
					$(e.currentTarget).toggleClass(options.selectedClass);
				}

				currentIndex = $.inArray(e.currentTarget, $icons.toArray());

				_prototype.dragging = dragging = true;
				_prototype.dragData = $icons.filter('.' + options.selectedClass).toArray();
				_prototype.dragDataSource = $container;

				createGhosts();
				resetScrollDistance();

				e.stopPropagation();
				e.preventDefault();
			});

			$container.on('click', '.' + options.iconClass, function (e) {
				if (e.ctrlKey && $icons.filter('.' + options.selectedClass).length >= 2)
					e.preventDefault(); // prevent open in new tab on ctrl + click
			});

			function dragFunc() {

				const offs_x2 = x2 - containerOffset.left + container.scrollLeft;
				const offs_y2 = y2 - containerOffset.top + container.scrollTop;

				let distances = [];
				let closestY = 0;

				self.update(false);

				for (let i = 0, n = $icons.length; i < n; ++i) {

					distances[i] = {
						x: centers[i].left - offs_x2,
						y: centers[i].top - offs_y2
					};

					if (Math.abs(distances[i].y) < Math.abs(distances[closestY].y))
						closestY = i;
				}

				const $el = $icons.eq(closestY);
				const minY = positions[closestY].top;
				const maxY = minY + $el.height();

				let indY = minY;
				let indH = $el.outerHeight();

				let noMatchRow = [];

				let leftX = null, rightX = null;

				for (let i = 0, n = distances.length; i < n; ++i) {

					noMatchRow[i] = (centers[i].top < minY || centers[i].top > maxY);
					if (noMatchRow[i]) continue;

					indY = Math.min(indY, positions[i].top);
					indH = Math.max(indH, $icons.eq(i).outerHeight());

					if (distances[i].x < 0)
						leftX = leftX != null ? Math.max(distances[i].x, leftX) : distances[i].x;
					else
						rightX = rightX != null ? Math.min(distances[i].x, rightX) : distances[i].x;
				}

				closestLeft = closestRight = -1;

				for (let i = 0, n = distances.length; i < n; ++i) {
					if (noMatchRow[i]) continue;

					if (distances[i].x == leftX)
						closestLeft = i;
					else if (distances[i].x == rightX)
						closestRight = i;
					if (closestLeft != -1 && closestRight != -1)
						break;
				}

				if ((currentIndex != closestLeft && currentIndex != closestRight) || !dragging) {

					reposition = true;

					const $el1 = $icons.eq(closestLeft);
					const $el2 = $icons.eq(closestRight);

					const pos1 = positions[closestLeft];
					const pos2 = positions[closestRight];

					let center;

					if (closestLeft == -1)
						center = pos2.left + (parseInt($el2.css('margin-left')) || 0);
					else if (closestRight == -1)
						center = pos1.left + $el1.outerWidth(true) - (parseInt($el1.css('margin-right')) || 0);
					else
						center = (pos1.left + pos2.left + $el2.outerWidth(true)) / 2;

					$indicator.css({
						left: center - $indicator.width() / 2 + 'px',
						top: indY + 'px',
						height: indH + 'px',
						'margin-top': $el1.css('margin-top')
					})

					if (options.indicatorHtml != '') $indicator.show();
				}
				else {
					reposition = false;
					$indicator.hide();
				}
			}

			let counter = 0;

			$element.on('dragenter', function (e) {
				counter++;
			});

			$element.on('dragleave', function (e) {
				counter--;
				if (counter == 0) {
					_prototype.currentElement = null;
					$indicator.hide();
				}
			});

			$element.on('dragover', function (e) {

				if (!_prototype.stopBubble && _prototype.currentElement != this) {
					_prototype.currentElement = this;
					self.update();
				}
				_prototype.stopBubble = true;

				x2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageX : e.pageX;
				y2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageY : e.pageY;

				if (_prototype.currentElement == this)
					console.log(y2 - containerOffset.top);

				canDropData = options.onDragOver.call(container, e.originalEvent.dataTransfer) == true;

				if (options.reorder && canDropData && $icons.length && _prototype.currentElement == this)
					dragFunc();
				else
					$indicator.hide();

				e.preventDefault();
			});

			$(document).on('dragover', function (e) {
				_prototype.stopBubble = false;
			});

			$element.on('drop', function (e) {

				if (canDropData) {

					const data = e.originalEvent.dataTransfer;
					options.onDropData.call(container, data);
					canDropData = false;
				}

				counter = 0;
				$indicator.hide();

				e.preventDefault();
				e.stopPropagation();
			});

			$element.on('mousedown touchstart', function (e) {

				self.update();

				$icons.removeClass(options.dropTargetClass);

				x1 = x2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageX : e.pageX;
				y1 = y2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageY : e.pageY;

				$element.focus();

				if (!e.ctrlKey)
					$icons.removeClass(options.selectedClass);

				selecting = true;

				resetScrollDistance();
				calculateRect();

				$rectangle.show();

				e.preventDefault();
				e.stopPropagation();
			});

			$element.on('mouseleave', function (e) {
				$indicator.hide();
				reposition = false;
				_prototype.currentElement = null;
			});

			$element.on('mousemove touchmove', function (e) {
				if (!_prototype.stopBubble && _prototype.currentElement != this) {
					_prototype.currentElement = this;
					self.update();
				}
				_prototype.stopBubble = true;

				if (_prototype.dragging) {
					if (hovering || _prototype.currentElement != element) {

						$indicator.hide();
						reposition = false;

					} else if (options.reorder && $icons.length) {

						dragFunc();
					}
				}
			});

			const maxSpeed = options.autoScrollSpeed;
			const play = options.autoScrollPlay;
			const gap = options.autoScrollGap;

			let timeout1 = null;
			let timeout2 = null;
			let speedY = 0;
			let speedX = 0;

			$container.on('mouseleave mousedown wheel DOMMouseScroll mousewheel keyup touchmove', function () {
				if (_prototype.dragging) clearTimeout(timeout1), clearTimeout(timeout2);
			});

			function stepUp() {
				if ((_prototype.dragging || selecting) && container.scrollTop > 0) {
					container.scrollTop -= Math.ceil(speedY);
					timeout1 = setTimeout(stepUp, 50 * (maxSpeed - speedY) / maxSpeed);
				}
			}
			function stepDown() {
				if ((_prototype.dragging || selecting) && container.scrollTop < container.scrollHeight - $container.innerHeight()) {
					container.scrollTop += Math.ceil(speedY);
					timeout1 = setTimeout(stepDown, 50 * (maxSpeed - speedY) / maxSpeed);
				}
			}
			function stepLeft() {
				if ((_prototype.dragging || selecting) && container.scrollLeft > 0) {
					container.scrollLeft -= Math.ceil(speedX);
					timeout2 = setTimeout(stepLeft, 50 * (maxSpeed - speedX) / maxSpeed);
				}
			}
			function stepRight() {
				if ((_prototype.dragging || selecting) && container.scrollLeft < container.scrollWidth - $container.innerWidth()) {
					container.scrollLeft += Math.ceil(speedX);
					timeout2 = setTimeout(stepRight, 50 * (maxSpeed - speedX) / maxSpeed);
				}
			}

			function calcSpeed1(distance) {
				return Math.min(1, distance / (play - gap)) * maxSpeed;
			}
			function calcSpeed2(distance) {
				return Math.min(1, distance / play) * maxSpeed;
			}

			// auto scroll feature :)
			$container.on('mousemove touchmove', function (e) {

				if (_prototype.dragging) {
					clearTimeout(timeout1), clearTimeout(timeout2);

					if (y2 < containerOffset.top + play) {
						speedY = calcSpeed1(play - (y2 - containerOffset.top));
						stepUp();
					}
					else if (y2 > containerOffset.top + $container.innerHeight() - play) {
						speedY = calcSpeed1(y2 - (containerOffset.top + $container.innerHeight() - play));
						stepDown();
					}
					if (x2 < containerOffset.left + play) {
						speedX = calcSpeed1(play - (x2 - containerOffset.left));
						stepLeft();
					}
					else if (x2 > containerOffset.left + $container.innerWidth() - play) {
						speedX = calcSpeed1(x2 - (containerOffset.left + $container.innerWidth() - play));
						stepRight();
					}
				}
			});

			$(document).on('mousemove touchmove', function (e) {

				_prototype.stopBubble = false;

				x2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageX : e.pageX;
				y2 = e.originalEvent.touches ? e.originalEvent.touches[0].pageY : e.pageY;

				if (selecting) {
					calculateRect();

					clearTimeout(timeout1), clearTimeout(timeout2);

					if (y2 < containerOffset.top + (parseInt($container.css('border-top-width')) || 0)) {
						speedY = calcSpeed2((containerOffset.top + (parseInt($container.css('border-top-width')) || 0)) - y2);
						stepUp();
					}
					else if (y2 > (containerOffset.top + $container.outerHeight() - 1)) {
						speedY = calcSpeed2(y2 - (containerOffset.top + $container.outerHeight() - 1));
						stepDown();
					}
					if (x2 < containerOffset.left + (parseInt($container.css('border-left-width')) || 0)) {
						speedX = calcSpeed2((containerOffset.left + (parseInt($container.css('border-left-width')) || 0)) - x2);
						stepLeft();
					}
					else if (x2 > (containerOffset.left + $container.outerWidth() - 1)) {
						speedX = calcSpeed2(x2 - (containerOffset.left + $container.outerWidth() - 1));
						stepRight();
					}
				}
				else if (_prototype.dragging) {

					if (dragging) { // are we dragging items belonging to self?

						if (!dragStarted) {
							dragStarted = true;
							$ghostContainer.show();
						}

						const mouseOffsetX = x2 - x1;
						const mouseOffsetY = y2 - y1;

						let bounds = $ghostContainer.data('bounds');

						bounds = {
							left: bounds.left + mouseOffsetX,
							top: bounds.top + mouseOffsetY,
							right: Math.min(bounds.right + mouseOffsetX, $(document).width()),
							bottom: Math.min(bounds.bottom + mouseOffsetY, $(document).height())
						}

						$ghostContainer.css({
							left: bounds.left + 'px',
							top: bounds.top + 'px',
							width: bounds.right - bounds.left + 'px',
							height: bounds.bottom - bounds.top + 'px'
						});
					}
				}

				if (e.type != 'touchmove')
					e.preventDefault();
			});

			$element.on('mouseup touchend', function (e) {

				if (_prototype.dragging) {
					_prototype.dragging = dragging = false; // shared across all instances!!

					if (!hovering && _prototype.dragData.length) {

						const $items = $(_prototype.dragData);

						const scrollTop = container.scrollTop;
						const scrollLeft = container.scrollLeft;

						if (_prototype.dragDataSource != $container) {

							$items = $(options.onBeforeMove.call(container, _prototype.dragData));

							if ($items.length) {

								if (!options.reorder || !$icons.length)
									$items.appendTo($container);
								else if (closestRight == -1)
									$items.insertAfter($icons.eq(closestLeft));
								else
									$items.insertBefore($icons.eq(closestRight));

								// optional resorting should take place here
								options.onMove.call(container, _prototype.dragData);
							}
						}
						else {

							if (!reposition) return;
							reposition = false;

							let offset = (closestLeft != -1 ? (closestLeft + 1) : (closestRight)) - currentIndex;
							if (offset > 0) offset--;
							if (offset == 0) return;

							let sequential = true;
							const icons = $icons.toArray();

							let j = $.inArray($items[0], icons) + 1;
							let n = $items.length;

							for (let i = 1; i < n; ++i, ++j) {
								if (j != $.inArray($items[i], icons)) {
									sequential = false;
									break;
								}
							}

							if (sequential && Math.abs(offset) < $items.length) {
								let $dropTarget;
								if (offset > 0) {
									$dropTarget = $items.last();
									for (let i = 0; i <= offset; ++i) {
										$dropTarget = $dropTarget.next();
										if (!$dropTarget.length)
											return;
									}
								}
								else {
									$dropTarget = $items.first();
									for (let i = 0; i > offset; --i) {
										$dropTarget = $dropTarget.prev();
										if (!$dropTarget.length)
											return;
									}
								}
								$items.insertBefore($dropTarget);
							}
							else {
								if (closestRight == -1)
									$items.insertAfter($icons.eq(closestLeft));
								else
									$items.insertBefore($icons.eq(closestRight));
							}

							options.onReposition($items.toArray());
						}

						// set scroll position back to original value
						container.scrollTop = scrollTop;
						container.scrollLeft = scrollLeft;

						self.update();
					}
				}
			});

			$(document).on('mouseup touchend', function (e) {

				_prototype.dragging = dragging = false;
				dragStarted = false;
				canDropItems = false;

				$rectangle.hide();
				$indicator.hide();
				destroyGhosts();

				if (selecting) {
					selecting = false;
					$icons.filter('.' + options.selectingClass)
						.addClass(options.selectedClass)
						.removeClass(options.selectingClass);
				}
			});

			function deleteCallback(items) {
				if (items.length) {
					$(items).remove();
					options.onDelete(items);
					self.update();
				}
			}

			$element.keydown(function (e) {

				switch (e.which) {
					case 27:// escape key
						if (selecting) {
							selecting = false;
							$icons.removeClass(options.selectedClass);
						}
						if (dragging) {
							_prototype.dragging = dragging = false; // shared across all instances!!
							dragStarted = false;
							destroyGhosts();
						}
						$rectangle.hide();
						$indicator.hide();
						break;

					case 46: // delete key
						const arr = $icons.filter('.' + options.selectedClass).toArray();
						if (arr.length) {
							options.onBeforeDelete(arr, deleteCallback);
						}
						break;

					case 65: // A key: select all
						if (e.ctrlKey) {
							$icons.addClass(options.selectedClass);
							e.preventDefault();
						}
						break;
				}
				e.stopPropagation();
			});
		}

		this.init();
	}

	const msie = document.documentMode;

	// shared vars
	$.desktopify.prototype.dragging = false;
	$.desktopify.prototype.dragData = [];
	$.desktopify.prototype.dragDataSource = $([]);
	$.desktopify.prototype.stopBubble = false;
	$.desktopify.prototype.currentElement = null;
	$.desktopify.prototype.supportsPointerEvents = (msie && msie <= 10) ? false : ('pointer-events' in document.body.style);

	console.log('pointer-events:', $.desktopify.prototype.supportsPointerEvents);

	$.fn.desktopify = function (options) {
		return this.each(function () {
			if (undefined == $(this).data('desktopify')) {
				const opts = $.extend({}, $.fn.desktopify.defaults, options);
				const plugin = new $.desktopify(this, opts);
				$(this).data('desktopify', plugin);
			}
		});
	}

	$.fn.desktopify.defaults = {
		element: '',
		exclude: '',
		iconClass: 'ui-icon',
		ghostClass: 'ui-ghost',
		selectedClass: 'ui-selected',
		selectingClass: 'ui-selecting',
		dropTargetClass: 'ui-drop-target',
		ghostContainerClass: 'ui-ghost-container',
		rectangleHtml: '<div class="ui-rectangle"/>',
		indicatorHtml: '<div class="ui-indicator"/>',
		autoScrollSpeed: 6,
		autoScrollPlay: 30,
		autoScrollGap: 5,
		ghostOffset: 10,
		reorder: true,
		onReposition: function () { },
		onBeforeDelete: function () { },
		onDelete: function () { },
		onBeforeHover: function () { return false },
		onHover: function () { },
		onBeforeMove: function () { return [] },
		onMove: function () { },
		onDragOver: function () { return false },
		onDropItems: function () { },
		onDropData: function () { }
	};
})(jQuery);
